;; Great Salt Lake
;;
;; Coded in 2017 by Lin Xiang; Last revised in 2021 by Lin Xiang (lxiang75@gmail.com; lin.xiang@uky.edu)
;;
;; If you mention this model in a publication, we ask that you include the citations below.
;;
;; Xiang, L. (2021). Great Salt Lake. Department of STEM Education, University of Kentucky, Lexington, KY.
;;
;;-----------------------------------------
;;CREATIVE COMMONS LICENSE
;;This code is distributed by Lin Xiang under a Creative Commons License:
;;Attribution-ShareAlike 4.0 International (CC BY-SA 4.0)
;;https://creativecommons.org/licenses/by-sa/4.0/
;;
;;-----------------------------------------

globals [sh gre]

breed [shrimps shrimp]
breed [salts salt]





to-report salinity
  report (pcolor - 99.5) / (-0.32)
end

to-report tolerable-habitate
  report count patches with [pcolor >= 93 and pcolor <= 97] / count patches
end

;===============================

to setup
ca
ask patches [set pcolor 95]
repeat 500
[
 Bearriver
 jordan-weber-river
 northarm
 evaporation
  diffuse pcolor 1
]

if show-shrimps? [

setup-shrimp
swim
]

set sh count shrimps

RESET-TICKS
end

;==================================

to go
 tick

 Bearriver
 jordan-weber-river
 northarm
 evaporation
 diffuse pcolor 1

ifelse show-shrimps?
  [
    swim
    tolerance
    setup-shrimp
    thinning
  ]
 [ask shrimps [die]]


crystalization

smooth-line

end

;--|Salinity|---------------------
to evaporation
  ask patches [ifelse pcolor >= 90
    [set pcolor (pcolor - ((Evaporation-Rate * 0.5) + 3) * 0.001) ]
    [set pcolor 89.9]
          ]
end


to Bearriver
  ask patch (max-pxcor * 0.8) (max-pycor)
  [if Bear-river > 0 [
       ask patches in-radius ((((Bear-river * 0.34) + 1) ^ 2) * 0.4)
    [set pcolor 99.5]
  ]
  ]
end

to jordan-weber-river
  ask patch (max-pxcor ) (min-pycor * 0.9)
  [if Jordan-and-Weber-Rivers > 0 [
       ask patches in-radius ((((Jordan-and-Weber-Rivers * 0.34) + 1) ^ 2) * 0.4)
    [set pcolor 99.5]
  ]]
end


to northarm
   ask patch (min-pxcor * 0.85 ) (max-pycor)
   [if North-Arm > 0
     [ask patches in-radius North-Arm
    [set pcolor 90.5]
   ]]
end

to crystalization
  ask patches [if pcolor < 90 [sprout-salts 1 [set color white set size 1 set shape "tile water"]]]
  ask salts [if pcolor >= 90 [die]]
end

;---|Brine Shrimp|--------------------------

to setup-shrimp
  ask patches with [pcolor >= 93 and pcolor <= 97]
[if not any? shrimps-here[
  sprout-shrimps 1 [
    set color 28
    set size 1
    set shape "b-shrimp-1"
    ]]]
end

to swim
  ask shrimps [
   rt random-float 45 fd 1
 ]
end


to tolerance
  ask shrimps [if abs (pcolor - 95) > 2 [die]]

end

to thinning
  ask patches [
    let num-here count shrimps-here
    if num-here > 1 [
      ask n-of (num-here - 1) shrimps-here [die]]]
end


;---|Supportive procedure|------------------
to smooth-line
  set sh sh * 0.9 + ( 1 - 0.9 ) * count shrimps
end
@#$#@#$#@
GRAPHICS-WINDOW
215
52
655
525
-1
-1
16.0
1
10
1
1
1
0
0
0
1
-13
13
-14
14
0
0
1
ticks
10.0

BUTTON
15
80
181
123
Set up/Reset Simulation
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
14
202
180
242
Run/Pause Simulation
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

SLIDER
658
53
691
274
Bear-river
Bear-river
1
10
4.0
1
1
NIL
VERTICAL

PLOT
741
58
986
186
Average Lake Salinity
Time
Salinity
0.0
10.0
0.0
30.0
true
false
"" ""
PENS
"default" 1.0 0 -8630108 true "" "plot mean [salinity] of patches"

SLIDER
438
529
655
562
Jordan-and-Weber-Rivers
Jordan-and-Weber-Rivers
1
10
4.0
1
1
NIL
HORIZONTAL

SLIDER
17
353
183
386
Evaporation-Rate
Evaporation-Rate
1
4
1.0
0.5
1
NIL
HORIZONTAL

MONITOR
741
197
939
246
Brine Shrimp Habitat ratio
tolerable-habitate
2
1
12

MONITOR
741
10
915
59
Average Lake Salinity
mean [salinity] of patches
1
1
12

SWITCH
18
402
184
435
Show-shrimps?
Show-shrimps?
0
1
-1000

PLOT
741
246
987
374
Shrimp Habitat Ratio
Time
Ratio
0.0
10.0
0.0
1.0
true
false
"" ""
PENS
"default" 1.0 0 -13791810 true "" "ifelse show-shrimps?\n[plot tolerable-habitate]\n[clear-plot]"

SLIDER
214
15
329
48
North-Arm
North-Arm
1
10
1.0
1
1
NIL
HORIZONTAL

MONITOR
742
389
881
438
Total of Shrimps
sh
0
1
12

PLOT
742
438
987
567
Brine Shrimp Population
Time
Shrimps
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"pen-1" 1.0 0 -955883 true "" "plot sh"
"pen-2" 1.0 0 -7500403 true "" "plot count shrimps"

TEXTBOX
15
252
198
322
Diffusion and evaporation take time. When you change a parameter, always allow 1-2 minutes for the lake to reach a new equilibrium (stable situation).
11
0.0
1

TEXTBOX
15
132
198
177
Allow 5~6 seconds to set up initial salinity and shrimp populations.\n
11
0.0
1

BUTTON
15
16
181
60
Restore Default Settings
set Evaporation-Rate 1\nset North-Arm 1\nset Bear-river 4\nset Jordan-and-Weber-Rivers 4\nset Show-shrimps? true\nsetup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

@#$#@#$#@
## WHAT IS IT?

This model simulates the salinity dynamics in the south arm of the Great Salt Lake caused by the surface inflows from three local rivers and the water exchanges with the north arm. The salinity changes, in turn, affect the brine shrimp population in the lake.

## DESIGN NOTES
1. Only three factors are included in the model to determine the salinity changes in the south arm of GSL: stream inflow, north arm water inflow, and evaporation.

2. Massive precipitation is unusual in the GSL area. When it happens, the stream inflow increases 

3. The salinity of the North arm is always high due to fewer freshwater inputs compared to the south arm.

4. To simply the model, only two river entrances, Bear River and Jordan-weber River, are simulated.

5. The highest salinity is 28% (saturation point) with color 90.5 and lowest is 0% (freshwater) with color 99.5.   { pcolor= (-0.32) * salinity + 99.5} When salinity is higher than 28%, crystallization starts.

6. The tolerable salinity for brine shrimps ranges from 8% (approximate color of 97) to 20% (approximate color of 93). 100 % of mortality rate is set to shrimps moving to the higher or lower saline area.

7. The brine shrimp population size constantly fluctuates due to salinity diffusion. The curves of shrimp populations have been smoothened to improve visualization.

Reference: 

Great Salt Lake Ecosystem Program. Brine Shrimp. https://wildlife.utah.gov/gsl/brineshrimp/salinity.php

White, J., Null, S. E., & Tarboton, D. G. (2014). More than meets the eye-managing salinity in Great Salt Lake, Utah. Lakeline Magazine, 34(3), 25.

Wurtsbaugh, W., Miller, C., Null, S., Wilcock, P., Hahnenberger, M., & Howe, F. (2016). Impacts of water development on Great Salt Lake and the Wasatch Front. https://qcnr.usu.edu/pdfs/publications/Great%20Salt%20Lake%20Water%20Level_Feb%2024%202016.pdf

## HOW TO USE IT

* First press on "set up/reset simulation" and then click on "run/pause simulation."
* Pull the bars on the sliders to adjust parameters while running the simulation.
* Turn on/off the switches to show or hide shrimps.


## THINGS TO NOTICE

1. When maximizing the surface inflows from all rivers, the mean salinity drops to around 6, representing the flooding scenario in the 1980s.

2. When minimizing the surface inflows from all rivers, the mean salinity increases to about 20.5. The temperature increase will further boost the mean.


## THINGS TO TRY

1. Change the inflows from the rivers and north arm and observe how it affects the 1) average salinity and salinity gradient of the lake and 2) population size and distribution of brine shrimps.

2. Change the evaporation rate and observe how it affects the 1) average salinity and salinity gradient of the lake and 2) population size and distribution of brine shrimps.


## CREDITS AND REFERENCES

This model is made by Dr. Lin Xiang at the University of Kentucky. If you mention this model in a publication, we ask that you include the citations below.

Xiang, L. (2021). Great Salt Lake. Department of STEM Education, University of Kentucky, Lexington, KY.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

b-shrimp
true
0
Polygon -7500403 true true 151 21 139 22 135 27 133 32 135 37 141 40 146 41 146 45 142 46 140 54 138 69 138 101 145 158 147 214 148 241 150 272 148 297 151 297 154 269 153 209 154 179 155 158 162 101 163 71 162 48 155 45 155 42 163 40 166 34 167 27 163 23 157 22
Circle -6459832 true false 162 20 8
Circle -6459832 true false 130 20 8
Polygon -7500403 true true 154 24 157 9 162 3 160 10 156 25
Polygon -7500403 true true 146 24 143 9 138 3 140 10 144 25
Polygon -7500403 true true 158 49 172 52 180 51 173 45 186 53 174 57 155 56
Polygon -7500403 true true 142 49 128 52 120 51 127 45 114 53 126 57 145 56
Polygon -7500403 true true 158 60 177 63 187 57 202 54 188 60 177 69 158 67
Polygon -7500403 true true 142 60 123 63 113 57 98 54 112 60 123 69 142 67
Polygon -7500403 true true 159 69 184 76 206 75 216 84 228 104 213 87 203 80 177 82 154 77
Polygon -7500403 true true 141 69 116 76 94 75 84 84 72 104 87 87 97 80 123 82 146 77
Polygon -7500403 true true 160 82 180 92 198 92 209 106 222 129 207 110 195 97 178 100 156 90
Polygon -7500403 true true 140 82 120 92 102 92 91 106 76 133 93 110 105 97 122 100 144 90
Polygon -7500403 true true 155 91 174 107 191 109 207 130 219 170 204 134 187 117 174 116 151 99
Polygon -7500403 true true 145 91 126 107 109 109 93 130 81 170 96 134 113 117 126 116 149 99
Polygon -7500403 true true 159 107 176 127 189 135 205 157 201 208 200 158 188 143 173 133 158 121
Polygon -7500403 true true 141 107 124 127 111 135 95 157 99 208 100 158 112 143 127 133 142 121
Polygon -7500403 true true 155 119 168 136 181 147 192 165 188 219 187 166 176 151 163 143 152 132
Polygon -7500403 true true 145 118 132 135 119 146 108 164 110 221 113 165 124 150 137 142 148 131
Polygon -7500403 true true 150 135 154 137 167 148 178 166 172 202 173 167 162 152 149 144 147 139
Polygon -7500403 true true 150 135 146 137 133 148 122 166 128 202 127 167 137 150 151 144 153 139
Polygon -7500403 true true 149 149 149 145 155 147 167 167 161 203 162 168 151 153 147 150 149 146
Polygon -7500403 true true 151 149 151 145 145 147 133 167 139 203 138 168 149 153 153 150 151 146
Line -6459832 false 150 31 150 232

b-shrimp-1
true
0
Polygon -7500403 true true 17 141 5 135 30 139 59 129 86 118 108 116 129 121 147 131 164 139 179 144 205 141 221 138 240 135 259 128 276 129 292 136 298 146 298 156 290 165 278 160 266 179 251 188 234 196 211 203 186 205 171 203 149 195 127 187 101 167 104 158 120 154 128 145 121 142 108 138 92 136 76 139 60 144 40 149 27 145 4 145
Polygon -7500403 true true 264 165 280 168 295 179 279 172 264 168 251 174 265 190 269 210 259 231 266 211 262 191 249 179 243 182 255 196 257 214 249 236 253 212 252 197 236 185 225 189 241 203 241 226 227 253 237 226 236 206 219 191 205 193 225 216 221 243 207 260 218 240 219 216 198 195 191 195 207 220 207 240 181 268 202 238 202 221 185 197 178 190 184 225 176 242 158 247 175 239 180 224 174 196 165 194 168 218 162 229 146 234 160 227 165 214 160 190 148 182 151 204 141 218 116 223 138 214 146 202 143 185 133 177 131 193 118 206 101 207 119 201 127 193 127 176 119 173 112 184 102 184 84 172 102 178 108 179 113 172 267 166
Polygon -7500403 true true 292 138 290 122 289 134
Polygon -16777216 true false 32 144 45 141 55 136 69 131 86 125 97 124 113 124 123 126 143 134 153 140 164 147 179 150 194 151 208 151 223 147 237 143 252 141 263 137 275 138 288 142 290 142 289 136 286 137 273 135 263 134 248 138 239 139 232 141 209 148 199 148 182 147 167 145 154 138 145 132 135 128 123 124 112 121 99 121 87 122 74 126 60 131 50 136 41 140 35 141 29 140 20 138

b-shrimp-2
true
0
Polygon -7500403 true true 150 16 164 22 170 34 170 46 156 56 157 62 168 71 174 82 177 148 165 189 151 197 144 198
Polygon -7500403 true true 151 16 136 22 130 34 130 46 144 56 143 62 132 71 126 82 123 148 135 189 149 197 156 198
Polygon -7500403 true true 154 22 162 6 164 7 156 22
Polygon -7500403 true true 146 22 138 6 136 7 144 22
Polygon -7500403 true true 142 68 130 64 123 53 121 36 119 51 127 68 137 77
Polygon -7500403 true true 158 68 170 64 177 53 179 36 181 51 173 68 163 77
Polygon -7500403 true true 165 84 178 75 187 56 205 47 189 59 179 80 163 91
Polygon -7500403 true true 164 88 197 84 213 94 221 107 209 95 195 89 165 95
Polygon -7500403 true true 170 99 203 103 219 113 227 126 215 114 201 108 170 106
Polygon -7500403 true true 170 114 207 123 223 133 231 146 219 134 205 128 172 123
Polygon -7500403 true true 168 129 201 137 217 147 225 160 213 148 199 142 168 137
Polygon -7500403 true true 154 140 187 148 203 158 211 171 199 159 185 153 154 148
Polygon -7500403 true true 157 155 175 160 191 170 199 183 187 171 173 165 154 162
Polygon -7500403 true true 162 159 173 172 180 182 181 206 177 189 174 180 160 168
Polygon -7500403 true true 151 166 162 179 169 189 164 214 164 195 163 187 149 175
Polygon -7500403 true true 135 84 122 75 113 56 95 47 111 59 121 80 137 91
Polygon -7500403 true true 136 88 103 84 87 94 79 107 91 95 105 89 135 95
Polygon -7500403 true true 130 99 97 103 81 113 73 126 85 114 99 108 130 106
Polygon -7500403 true true 130 114 93 123 77 133 69 146 81 134 95 128 128 123
Polygon -7500403 true true 132 129 99 137 83 147 75 160 87 148 101 142 132 137
Polygon -7500403 true true 146 140 113 148 97 158 89 171 101 159 115 153 146 148
Polygon -7500403 true true 143 155 125 160 109 170 101 183 113 171 127 165 146 162
Polygon -7500403 true true 138 159 127 172 120 182 119 206 123 189 126 180 140 168
Polygon -7500403 true true 149 166 138 179 131 189 136 214 136 195 137 187 151 175
Polygon -7500403 true true 147 190 147 212 147 226 146 250 147 269 146 290 147 298 149 283 155 298 154 277 153 256 153 246 154 227 154 208 153 200 153 191

b-shrimp-3
true
0
Polygon -7500403 true true 151 17 139 18 135 23 133 28 135 33 141 36 146 37 146 41 142 42 140 50 138 65 138 97 145 154 151 201 143 233 155 255 138 271 146 270 160 254 149 232 156 201 155 154 162 97 163 67 162 44 155 41 155 38 163 36 166 30 167 23 163 19 157 18
Circle -6459832 true false 162 20 8
Circle -6459832 true false 130 20 8
Polygon -7500403 true true 154 24 157 9 162 3 160 10 156 25
Polygon -7500403 true true 146 24 143 9 138 3 140 10 144 25
Polygon -7500403 true true 158 49 172 52 180 51 173 45 186 53 174 57 155 56
Polygon -7500403 true true 142 49 128 52 120 51 127 45 114 53 126 57 145 56
Polygon -7500403 true true 158 60 177 63 187 57 202 54 188 60 177 69 158 67
Polygon -7500403 true true 142 60 123 63 113 57 98 54 112 60 123 69 142 67
Polygon -7500403 true true 159 69 184 76 206 75 216 84 228 104 213 87 203 80 177 82 154 77
Polygon -7500403 true true 141 69 116 76 94 75 84 84 72 104 87 87 97 80 123 82 146 77
Polygon -7500403 true true 160 82 180 92 198 92 209 106 222 129 207 110 195 97 178 100 156 90
Polygon -7500403 true true 140 82 120 92 102 92 91 106 76 133 93 110 105 97 122 100 144 90
Polygon -7500403 true true 155 91 174 107 191 109 207 130 219 170 204 134 187 117 174 116 151 99
Polygon -7500403 true true 145 91 126 107 109 109 93 130 81 170 96 134 113 117 126 116 149 99
Polygon -7500403 true true 159 107 176 127 189 135 205 157 201 208 200 158 188 143 173 133 158 121
Polygon -7500403 true true 141 107 124 127 111 135 95 157 99 208 100 158 112 143 127 133 142 121
Polygon -7500403 true true 155 119 168 136 181 147 192 165 188 219 187 166 176 151 163 143 152 132
Polygon -7500403 true true 145 118 132 135 119 146 108 164 110 221 113 165 124 150 137 142 148 131
Polygon -7500403 true true 150 135 154 137 167 148 178 166 172 202 173 167 162 152 149 144 147 139
Polygon -7500403 true true 150 135 146 137 133 148 122 166 128 202 127 167 137 150 151 144 153 139
Polygon -7500403 true true 149 149 149 145 155 147 167 167 161 203 162 168 151 153 147 150 149 146
Polygon -7500403 true true 151 149 151 145 145 147 133 167 139 203 138 168 149 153 153 150 151 146

b-shrimp-4
true
0
Polygon -7500403 true true 151 21 139 22 135 27 133 32 135 37 141 40 146 41 146 45 142 46 138 54 121 83 117 114 130 149 145 166 148 241 150 272 148 297 151 297 154 269 153 209 156 166 172 149 184 112 180 86 165 53 155 45 155 42 163 40 166 34 167 27 163 23 157 22
Circle -6459832 true false 162 20 8
Circle -6459832 true false 130 20 8
Polygon -7500403 true true 154 24 157 9 162 3 160 10 156 25
Polygon -7500403 true true 146 24 143 9 138 3 140 10 144 25
Polygon -7500403 true true 158 49 172 52 180 51 173 45 186 53 174 57 155 56
Polygon -7500403 true true 142 49 128 52 120 51 127 45 114 53 126 57 145 56
Polygon -7500403 true true 159 67 178 70 188 64 203 61 189 67 178 76 159 74
Polygon -7500403 true true 142 67 123 70 113 64 98 61 112 67 123 76 142 74
Polygon -7500403 true true 159 82 184 89 206 88 216 97 228 117 213 100 203 93 177 95 154 90
Polygon -7500403 true true 141 82 116 89 94 88 84 97 72 117 87 100 97 93 123 95 146 90
Polygon -7500403 true true 161 102 181 112 199 112 210 126 223 149 208 130 196 117 179 120 157 110
Polygon -7500403 true true 140 101 120 111 102 111 91 125 76 152 93 129 105 116 122 119 144 109
Polygon -7500403 true true 154 118 158 120 171 131 182 149 176 185 177 150 166 135 153 127 151 122
Polygon -7500403 true true 148 118 144 120 131 131 120 149 126 185 125 150 135 133 149 127 151 122
Line -6459832 false 150 31 150 232

bird
true
0
Polygon -7500403 true true 154 72 150 62 147 72 139 77 135 86 129 98 117 98 109 95 104 89 99 78 90 74 74 77 56 85 40 97 30 112 0 150 42 135 74 126 87 138 109 148 130 154 134 163 135 165 105 209 135 224 152 228 165 224 195 209 165 165 168 163 172 153 193 148 213 138 227 128 259 135 300 149 271 113 261 99 249 86 230 78 210 74 203 78 196 92 185 95 171 97 166 85 162 76

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

grebe
false
0
Polygon -6459832 true false 64 70 14 100 72 90 67 77 60 90 69 79
Polygon -7500403 true true 58 75 68 52 90 37 108 35 125 43 133 58 118 62 124 74 125 91 116 112 110 126 102 141 96 156 96 168 107 164 137 148 163 139 196 132 170 148 208 134 238 136 198 154 239 142 258 139 292 155 253 156 228 163 258 160 282 160 261 201 237 205 204 208 166 212 135 217 95 223 60 225 40 229 37 218 36 203 45 180 61 157 75 138 82 125 87 108 87 97 80 93 69 91 58 90 58 80
Polygon -2674135 true false 68 66 64 71 65 77 72 79 76 73 73 67
Polygon -955883 true false 73 68 95 49 118 47 138 59 122 59 130 70 116 67 122 74 129 82 131 97 113 72 120 89 123 97 122 112 114 95 107 83 106 99 99 83 91 77 78 75
Polygon -6459832 true false 105 202 115 181 151 168 186 161 224 160 252 159 281 161 261 201 241 204 210 207 183 210 159 213 132 217 116 218

grebe-1
false
0
Polygon -6459832 true false 64 70 14 100 72 90 67 77 60 90 69 79
Polygon -7500403 true true 58 75 68 52 90 37 108 35 125 43 133 58 118 62 124 74 125 91 116 112 110 126 102 141 96 156 96 168 107 164 137 148 163 139 196 132 170 148 208 134 238 136 198 154 239 142 258 139 292 155 253 156 228 163 258 160 282 160 261 201 237 205 204 208 166 212 135 217 95 223 60 225 40 229 37 218 36 203 45 180 61 157 75 138 82 125 87 108 87 97 80 93 69 91 58 90 58 80
Polygon -2674135 true false 68 66 64 71 65 77 72 79 76 73 73 67
Polygon -955883 true false 73 68 95 49 118 47 138 59 122 59 141 75 128 73 121 71 132 84 141 98 120 84 123 91 125 100 123 111 114 95 107 83 106 99 99 83 91 77 78 75
Polygon -6459832 true false 125 212 138 208 158 199 185 194 218 188 240 180 281 161 258 187 241 199 210 202 183 206 159 210 132 217 116 218

gull
true
0
Polygon -7500403 true true 154 73 150 63 147 73 139 78 135 87 129 99 117 99 109 96 104 90 99 79 90 75 74 78 56 86 40 98 30 113 0 151 42 136 74 127 87 139 109 149 127 155 132 173 138 188 122 210 137 218 151 221 166 218 178 209 164 190 170 175 175 155 193 149 213 139 227 129 259 136 300 150 271 114 261 100 249 87 230 79 210 75 203 79 196 93 185 96 171 98 166 86 162 77
Polygon -16777216 true false 190 148 209 139 222 128 228 124 262 134 294 146 270 119 259 104 246 92 225 82 244 99 251 108 230 116 208 132
Polygon -2674135 true false 151 60 145 79 151 71 156 78
Polygon -16777216 true false 110 148 91 139 78 128 72 124 38 134 6 146 30 119 41 104 54 92 75 82 56 99 49 108 70 116 92 132
Polygon -16777216 true false 160 79 162 86 155 77
Polygon -16777216 true false 140 79 138 86 145 77

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tile log
false
0
Rectangle -7500403 true true 0 0 300 300
Line -16777216 false 0 30 45 15
Line -16777216 false 45 15 120 30
Line -16777216 false 120 30 180 45
Line -16777216 false 180 45 225 45
Line -16777216 false 225 45 165 60
Line -16777216 false 165 60 120 75
Line -16777216 false 120 75 30 60
Line -16777216 false 30 60 0 60
Line -16777216 false 300 30 270 45
Line -16777216 false 270 45 255 60
Line -16777216 false 255 60 300 60
Polygon -16777216 false false 15 120 90 90 136 95 210 75 270 90 300 120 270 150 195 165 150 150 60 150 30 135
Polygon -16777216 false false 63 134 166 135 230 142 270 120 210 105 116 120 88 122
Polygon -16777216 false false 22 45 84 53 144 49 50 31
Line -16777216 false 0 180 15 180
Line -16777216 false 15 180 105 195
Line -16777216 false 105 195 180 195
Line -16777216 false 225 210 165 225
Line -16777216 false 165 225 60 225
Line -16777216 false 60 225 0 210
Line -16777216 false 300 180 264 191
Line -16777216 false 255 225 300 210
Line -16777216 false 16 196 116 211
Line -16777216 false 180 300 105 285
Line -16777216 false 135 255 240 240
Line -16777216 false 240 240 300 255
Line -16777216 false 135 255 105 285
Line -16777216 false 180 0 240 15
Line -16777216 false 240 15 300 0
Line -16777216 false 0 300 45 285
Line -16777216 false 45 285 45 270
Line -16777216 false 45 270 0 255
Polygon -16777216 false false 150 270 225 300 300 285 228 264
Line -16777216 false 223 209 255 225
Line -16777216 false 179 196 227 183
Line -16777216 false 228 183 266 192

tile stones
false
0
Polygon -7500403 true true 0 240 45 195 75 180 90 165 90 135 45 120 0 135
Polygon -7500403 true true 300 240 285 210 270 180 270 150 300 135 300 225
Polygon -7500403 true true 225 300 240 270 270 255 285 255 300 285 300 300
Polygon -7500403 true true 0 285 30 300 0 300
Polygon -7500403 true true 225 0 210 15 210 30 255 60 285 45 300 30 300 0
Polygon -7500403 true true 0 30 30 0 0 0
Polygon -7500403 true true 15 30 75 0 180 0 195 30 225 60 210 90 135 60 45 60
Polygon -7500403 true true 0 105 30 105 75 120 105 105 90 75 45 75 0 60
Polygon -7500403 true true 300 60 240 75 255 105 285 120 300 105
Polygon -7500403 true true 120 75 120 105 105 135 105 165 165 150 240 150 255 135 240 105 210 105 180 90 150 75
Polygon -7500403 true true 75 300 135 285 195 300
Polygon -7500403 true true 30 285 75 285 120 270 150 270 150 210 90 195 60 210 15 255
Polygon -7500403 true true 180 285 240 255 255 225 255 195 240 165 195 165 150 165 135 195 165 210 165 255

tile water
false
0
Rectangle -7500403 true true -1 0 299 300
Polygon -11221820 true false 105 259 180 290 212 299 168 271 103 255 32 221 1 216 35 234
Polygon -11221820 true false 300 161 248 127 195 107 245 141 300 167
Polygon -11221820 true false 0 157 45 181 79 194 45 166 0 151
Polygon -11221820 true false 179 42 105 12 60 0 120 30 180 45 254 77 299 93 254 63
Polygon -11221820 true false 99 91 50 71 0 57 51 81 165 135
Polygon -11221820 true false 194 224 258 254 295 261 211 221 144 199

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.2.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
