;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;; VARIABLE DECLARATIONS ;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
extensions [array]
globals [
  x y                               ; x and y coordinates, in meters
  month                             ; integer representing the month (from 0-12)
  month-name                        ; array of month names, from JAN-DEC
  temperature                       ; average water temperature, in C
  initial-extent-eelgrass           ; count of how many seedlings we planted
  max-extent-eelgrass               ; the maximum extent of the meadow (largest veg-mass amount)
  seed-germination                  ; the probability of seed germination and survival (decimal)
  simulation-started                ; alerts the free draw mode that the simulation is actively running
]
breed [eelgrasses eelgrass]         ; each eelgrass turtle is a clump of eelgrass representing many shoots
breed [green-turtles green-turtle]  ; each green-turtle turtle is a literal sea turtle
eelgrasses-own [
  veg-mass                          ; the biomass of eelgrass shoots, in gC/m2
  rhizome-mass                      ; the biomass of eelgrass root & rhizomes, in gC/m2
  epiphyte                          ; the biomass of epiphyte growth, in gC/m2
  age                               ; the birth month (planting month) of the eelgrass seedling (tick)
  seeds                             ; the maximum biomass of the eelgrass clump (used for computing seeds)
]
green-turtles-own [energy]          ; energy represents the turtles' stomach and fat stores
turtles-own []
patches-own [
  depth                             ; mean water depth, neglecting tides [Wetzel (1986) found no tidal impact]
]



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;; MODEL SETUP ;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
to setup
  clear-all                         ; reset everything to NetLogo default
  reset-ticks                       ; reset ticks to 0
  set simulation-started false      ; alerts free draw mode that we're in setup mode

  ;; Set up the calendar and initial water temperature
  set month 10       ;Calendar month in ordinal numbers: 10=October.
  set temperature 16.25 - 13.75 * (cos(360 * (month + .5) / 12 - 25)) ; from Buzzelli et al (1999)
  set temperature temperature + Global-Warming ;for experiment
  set month-name array:from-list [ "JAN" "FEB" "MAR" "APR" "MAY" "JUN" "JUL" "AUG" "SEP" "OCT" "NOV" "DEC" ]
  Display-month-year                ; Display the month and year so we can see it on the setup screen

  ;; Set the contour of the seafloor (water depth)
  ask patches [
    setup-depth
  ]

  ;; All new turtles will be eelgrass shape unless otherwise specified
  set-default-shape eelgrasses "eelgrass"

  ;; Plant the initial crop of eelgrass in the pattern specified by the pick-list
  plant-initial-eelgrass-pattern

  ;Remember how much eelgrass we planted so that we can tell the % increase in growth
  set initial-extent-eelgrass count eelgrasses        ; # of initial seedlings
  set max-extent-eelgrass initial-extent-eelgrass     ; the initial max = initial seedlings
end



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;; MODEL GO ;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
to go
  ;This model represents the passage of time for a small section of the Chesapeake Bay.
  ;Each tick represents one month in time.  The water temperature changes per month.
  set month month + 1
  if month > 12 [set month 1]
  ;set the water temperature depending on which month it is
  set temperature 16.25 - 13.75 * (cos(360 * (month + .5) / 12 - 25)) ; from Buzzelli et al (1999)
  Display-Month-Year

  ;Alert free draw mode that the simulation has started so that it will stop resetting ticks to 0
  set simulation-started true

  ;Don't start the model if the user didn't plant any eelgrass
  if ticks = 0 and count eelgrasses = 0 [
    user-message ["You must plant some eelgrass, either by using a preset pattern, or by using free plant mode and manually picking a pattern."]
    stop
  ]

  ;; Model the migration of green turtles
  ;; A user-specified number of green turtles will migrate into this area every spring
  if month = 5 [
    ;Green Turtles migrate into the bay in late spring (May)
    set-default-shape green-turtles "turtle"
    create-green-turtles migrating-green-turtles  ;; create the turtles, then initialize their variables
    [
      set color black
      set size 1.5               ;; easier to see, though a bit exaggerated in size
      set energy 4               ;; Assume each turtle has some fat and has eaten a recent meal
      move-to one-of eelgrasses  ;; Seek out the nearest eelgrass to eat
    ]
  ]
  ;; Green turtles migrate out of the bay in mid-Sept (deleted from the model in Oct)
  if month = 10 [
    ask green-turtles [die]   ; All green turtles either leave or die
  ]

  ;; Model the growth of eelgrass
  ;; Use Jarvis et al (2014), Buzzelli et al (1999) and Wetzel & Neckles (1986) model equations,
  ;; adapted for agent-based modeling.
  ask eelgrasses [
    grow-eelgrass                ; metabolic growth demon
    rhizome-eelgrass             ; rhizone reproduction demon
  ]


  ;; Model the behavior of green turtles based on a simple grazing model
  ;; Turtles will seek out the closest eelgrass and eat it
  repeat 30 [                    ; repeat each day for each month (30 times per tick)
    ask green-turtles [
      move                       ; movement demon (move to eelgrass)
      set energy energy - 1      ; metabolism requires energy (digestion & fat stores)
      eat-eelgrass               ; grazing demon
      death                      ; mortality demon
    ]
  ]

  ;If this was the new seasonal high for eelgrass growth, remember it
  let present-extent-eelgrass sum [veg-mass] of eelgrasses
  if present-extent-eelgrass > max-extent-eelgrass [
    set max-extent-eelgrass present-extent-eelgrass
  ]

  ;Release seeds in late summer; model as released in Oct so that
  ;the seeds will germinate in Nov.
  if month = 10 [
    ; Equation from Jarvis et al(2014) adapted assuming 3cm seed burial and 1.25% SOC (.
    ; Seed-germination represents the percent chance of seed germinating and surviving
    ; Seed-survival is a user-entered base probability of seed survival
    set seed-germination seed-survival * 0.4 / (1 + e ^ ( -0.1432 + (1.1261 * 0.03) - (1.3964 * .0125)) )
    ask eelgrasses [
      seed-eelgrass              ; seed reproduction demon
    ]
  ]

  tick
  if ticks = Months-of-experiment [stop] ;stop after a user-input number of months, usually 60-72 months

  ;stop if all the eelgrasses are dead or eaten
  if count eelgrasses = 0 [
    user-message ("All the eelgrass has died or been eaten.  Very sad.  Fortunately, this was just a computer simulation!")
    stop
  ]
end



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;; EELGRASS GROWTH ;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
to grow-eelgrass
  ;; This subroutine models the growth of a particular groups of shoots of eelgrass represented by each turtle
  ;; This growth is based upon models by Verhagen et al and Jarvis et al (2014)
  ;; and varies by month, depth (vs plant height), water temperature, and water turbidity
  ;;
  ;; The maximum size of a shoot in this model is 1m, and is tracked by the turtle size property.
  ;; For simplification purposes, biomass is modeled as proportionate to size.

  ;Jarvis et al (2014) equations for Z. Marina (based on Buzzelli):
  ;vegatative biomass: Czms=Czms(t-dt)+(Pzms+Tczmss-Mzms-Rzms-Td)dt
  ;rhizome biomass: Czmr=Czmr(t-dt)+(Td+Tczmsr-Mzmr-Rzmr)dt
  ;Td = translocation down from shoots to roots = Tzms
  ;  Tzms = 0.3 (unitless)
  ;Pzms = Z. marina shoot production = unknown
  ;Tczmss = transfer of seedling biomass to vegetative shoot biomass
  ;Tczmsr = transfer of seedling biomass to rhizome biomass
  ;Mzms = shoot mortality = MRzms = 0.007/day
  ;Mzmr = rhizome mortality = MRzmr = 0.000085/day (jan-jun) or 0.0095/day (jul-dec)
  ;Rzms = shoot respiration
  ;Rzmr = root and rhizome respiration, Arrhenius relationship at optimal 20C
  ;  RRzmr = root respiration at 20C = 0.0005/day
  ;  THETAzmr = root and rhizome Arrhenius constant = 1.25 (unitless)
  ;Td = translocation down, shoot-to-rhizome transfer = Tzms
  ;Tzms = Z. marine shoot-to-root transfer = 0.3 (unitless)
  ;Note: Jarvis calculated mortality as a fraction in the biomass model, but we will calculate it
  ;probabilistically and separately.

  ;Buzzelli et al (1999) equations for Z. Marina (based on Wetzel):
  ;Czs(t) = shoot carbon
  ;       = Czs(t-dt) + (Pzs - Rzs - Mzs - Tzc)dt
  ;Czr(t) = rhizome carbon
  ;       = Czr(t-dt) + (Tzc - Rzrr - Mzrr)dt
  ;Tzc = translocation, shoot to rhizome
  ;Pmzm = maximum photosynthesic rate
  ;     = (0.0025*T + 0.0049) * (1 - (Tw-25)/10) per day
  ;  Tw = water temperature, approximated seasonally using Wetzel's sinusoidal model (1986)
  ;Rzs = shoot respiration
  ;    = PRzs*(0.00317*(Tw+0.105)+e(0.135*Tw-10.1)) per day
  ;Rzrr= root and rhizome respiration, Arrhenius relationship at optimal 20C
  ;    = Rkzr & THETAzr ^ (Tw-Topt)
  ;    = 0.0005 * 1.25 ^ (Tw - 20)
  ;  Rkzr = root respiration at 20C = 0.0005/day
  ;  THETAzmr = root and rhizome Arrhenius constant = 1.25 (unitless)
  ;FBzsc = feedback relationship between maximum and limiting shoot biomass levels
  ;      = (Czs - Climzs) / (Cmaxzs - Climzs)
  ;      = (Czs - 100) / 100
  ;Czs   = shoot biomass
  ;Climzs = 100
  ;Cmaxzs = 200
  ;Does Td = 0.3*FBzsc ?? Buzzelli describes this but doesn't explicitely show it
  ;Cepi = Epiphyte biomass
  ;     = Depi(t-dt) + (Pepi - Repi - Mepi - Gepi)dt
  ;Pij = Photosynthesis rate coefficient
  ;    = Pmax(t)*[PARvp / (Ik' + PARvp)]
  ;Gepi= grazing
  ;Pmepi=epiphyte photosynthesis
  ;     =0.0091 * Tw * [1.0 - (Tw-25)/20)]
  ;PARzm=photosynthetically active radiation reaching the eelgrass
  ;     =PARh * (1 - (.75 * sqrt(Cepi/Czs/2.9) ) )
  ;PARh = depth corrected for habitat
  ;     = PAR0 * e^(-kd*hhab)
  ;   kd= light extinction coefficient, function of chlorophyll, POC and DOC
  ;     = 0.04
  ; hhab= habitat depth (m)

  ;Wetzel & Neckles (1986) equations for Z. Marina (based on Wiegart):
  ;Fij = Carbon transfer from CO2 into eelgrass leaves, general form
  ;    = Pij*Xj*[1-FBij)(1-(GBjj*Cjj))]
  ; Xj = Leaf biomass
  ; Cij= metabolic correction rate
  ;    = 1.0 - (Rj / Pij)
  ;Pmax= linear statistically fitted function represented light-saturated rate at tempereature (t)
  ;PARvp = photosynthetically active radiation, or intensity of light reaching the leaves
  ;Ik' = half-saturation PAR intensity, based upon assumption that fresh new leaves were epiphyte-free
  ;FBij=non-linear feedback function of CO2 concentration
  ;    =1 - [(Xi - Gij) / (Aij - Gij)]
  ; Xi =CO2 concentration
  ; Aij=CO2 concentration below which carbon became limiting
  ; Gij=CO2 concentration at which photosynthesis by eelgrass became zero
  ;FBjj=non-linear feedback function of metabolic limitations (crowding, nutrients, etc.)
  ;NOTE: Wetzel assumed Td was 17% of net leaf organic matter production, reduced
  ;when either above or below-ground compartments became limiting.
  ;Wetzel found epiphye:eelgrass biomass ratio was 0.04 during early spring and 1.5 during the late-summer decline
  ;PAR reaching the plant was reduced 22% due to epiphytes and another 10% due to epiphyte-limited CO2 diffusion.
  ;Maximum density was set at 150gC/m2 in the density feedback as this is the densest observed in the Chesapeake

  ;; Programming note: these calibration levels worked well during model validation.
  ;Td-Calibration = 4.1
  ;Rzms-Calibration = 1.0
  ;Rzmr-Calibration = 3.3
  ;mortality-calibration = 0.1
  ;max-clumps-per-patch = 10
  ;Rhizome-shoot-distance = .50

  ;;Eelgrass metabolic equations
  let PARzm  e ^ (-0.4 * ([depth] of patch-here)) * (1 - .32) ;represents light decrease from epiphytes and depth, from Wetzel 1986.
  let Pzms PARzm * (0.0025 * (temperature - 10 ) + 0.0049) * (1 - (temperature - 35) / 10)   ;adapted from Buzzelli et al (1999)
  if Pzms < 0 [set Pzms 0]
  let Tczmss  0 ;transfer of seedling biomass to vegetative shoot biomass
  let Rzms  Pzms * veg-mass * Rzms-calibration * (0.00317 * (temperature + 0.105) + e ^ (0.135 * temperature - 10.1)) ;from Buzzelli et al (1999)  *Pzms??
  ;if temperature <= 14 [set Rzms 0] ;Jarvis et al 2014: reduce respiration to 0 below 14C.
  let Rzmr  0.0005 * rhizome-mass * Rzmr-calibration * (1.25 ^ (temperature - 20) + 2)  ;adapted from Buzzelli et al (1999)
  let MRzms  0.0095   ; (jul-dec)
  if month < 7 [set MRzms  0.000085] ; (jan-jun)
  set MRzms MRzms * mortality-calibration

  let Tflower 0                                                               ;Initially assume no flower/seed production
  if month >= 6 and temperature < 21 and age > 12 [set Tflower veg-mass * .1] ;Calculate flower / seed production for flowering plants

  ;Td = translocate down biomass from shoots to rhizome
  let Td .3 * Td-calibration * (veg-mass) / 100 ;adapted from Jarvis et al (2014) and Buzzelli et al (1999)
  if Td < 0 [set Td 0] ;no translocation up, just translocation down
  if veg-mass > 0 or month < 7  [ ;Inhibit regrowth of died-off shoots in late summer/fall
    set veg-mass veg-mass + 30 * (Pzms - Rzms - Tflower) ; adapted Jarvis et al (2014) equation
  ]
  if veg-mass < 0 [set veg-mass 0] ; cannot have negative vegetative mass
  ifelse (veg-mass > (30 * Td))    ; if there is shoot biomass to translocate into the rhizome
    [ set veg-mass veg-mass - 30 * Td]    ;translocate shoot biomass into the rhizome if that biomass exists
    [ set Td veg-mass / 30                ;translocate all remaining shoot biomass to translocate
      set veg-mass 0]
  set rhizome-mass rhizome-mass + 30 * (Td - Rzmr) ;  Jarvis et al (2014) equation

  ;let calibration-constant .1 ;converts biomass equations to model clumps - not yet fully implemented

  ;Check for random mortality based on Jarvis et al values
  if random-float 1 < MRzms * 30 [die]


  ;Estimate the amount of seeds as proportionate to the maximum vegetative mass of the eelgrass.
  if veg-mass > seeds and (ticks - age) > 12 [set seeds veg-mass]

  ;Render the visual model based on shoot biomass, but not smaller than size 0.5.
  set color green - 2
  ifelse veg-mass > 1.5
    [set size veg-mass / 3]
    [set size .5]
end



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;; EELGRASS RHIZOME GROWTH ;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
to rhizome-eelgrass
  ; Simulates new rhizome reproduction sprouting at a random point near the parent shoot
  ; A new sprout is hatched if the plant is mature enough and the immediate area is not already overcrowded
  if ((veg-mass > .8) or (veg-mass > .8 * [depth] of patch-here)) and count eelgrasses-here <= Max-clumps-per-patch  [
    hatch-eelgrasses 1 [                                       ; create a new clone of this clump
      set veg-mass 0                                           ; Starts off with no shoots (yet)
      set epiphyte 0                                           ; No epiphtye without shoots to grow on
      set rhizome-mass .01                                     ; Model a small piece of rhizome as a new plant
      set heading random 360                                   ; Sets the random direction the new clump will be
      forward (random-float (Rhizome-shoot-distance) + Rhizome-shoot-distance / 2) ;sets the random distance the new clump will be
      set size .1                                              ; Render as a very small dot
      set heading 0                                            ; Reset heading to 0 so that the eelgrass turtle looks like grass when it renders
    ]
    if rhizome-mass > 0.01 [set rhizome-mass rhizome-mass - 0.01] ;This rhizome mass just transferred to the new clump.
  ]
end



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;; EELGRASS SEED PROPAGATION ;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
to seed-eelgrass
  ;; algorithm model for ellgrass sexual reproduction through seeds
  ; Flowers and seeds grow between may/june and sept.  (http://seagrant.mit.edu/eelgrass/eelgrassscience/biology.html)
  ; germination is initiated below 20C, typically in November.
  ; Floating flowering shoots spread seeds throughout the bay.
  ; Eelgrass in the Chesapeake doesn't flower the first year (Jarvis et al, 2014).
  ; 10% of shoot density flowers when it's <21C in June.
  ; .33 seed predation rate

  ;The seeds variable stored information about vegetative mass of flower-producing shoots
  ;if seeds > 0, then this shoot flowered.
  if seeds > 0 [
    set seeds seeds / 3 ;reduce # of seeds due to predation (Jarvis et al, 2014)

    ;Estimate the probability of seed germination based on # of seeds in the clump
    ;The individual germination rate is very low but is multiplied by the # of seeds
    ;in each clump.
    if random-float (1) > (1 - seed-germination ^ seeds) [
      ifelse random (100) < 5
        ;Assume 5% of seeds are distributed purely randomly
        [ set x random (max-pxcor * 2 ) + min-pxcor
          set y random (max-pycor * 2 ) + min-pycor
          ;If this seed did not land in an overpopulated area, create a new eelgrass clump
          if count eelgrasses-at x y <= Max-clumps-per-patch  and x <= max-pxcor [
            hatch-eelgrasses 1 [
              set veg-mass 0
              set epiphyte 0
              set rhizome-mass .01
              set xcor x
              set ycor y
              set size .1
              set heading 0
            ]
          ]
        ]
        ;Assume prevailing rising tides normally carry seeds a logarithmic distance to the left
        [ let seed-angle 90 + random-float (10) - 5     ;Assume seeds are carried by the rising tide in the direction seed-angle
          let seed-distance e ^ random-float (5)        ;Seeds drop to the bottom after a random distance, seed-distance
          set x xcor + seed-distance * cos (seed-angle) ;Convert polar coordinate to (x,y)
          set y ycor + seed-distance * sin (seed-angle) ;Convert polar coordinate to (x,y)

          ;If this seed did not land in an overpopulated area or outside of the model area, create a new eelgrass clump
          if count eelgrasses-at x y <= Max-clumps-per-patch  and x <= max-pxcor [
            hatch-eelgrasses 1 [
              set veg-mass 0
              set epiphyte 0
              set rhizome-mass .01
              set heading seed-angle
              forward seed-distance
              set size .1
              set heading 0
            ]
          ]
        ]
    ]
  ]
end



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;; PLANT EELGRASS ;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
to plant-eelgrass
  create-eelgrasses 1 ;plant an eelgrass clump
  [
    set color yellow       ;; Color it yellow for visualization so that it's easy to see
    set shape "eelgrass"   ;; Make it eelgrass-shaped
    set veg-mass 0         ;; Starts off with no grass
    set rhizome-mass .01   ;; Starts off with a small amount of seedling/rhizome mass
    set size 1.5           ;; Exaggerate the size for visualization so that it's easy to see
    setxy x y              ;; Place the eelgrass at the coordinates x,y (global variables set before calling this subroutine)
    set heading 0          ;; Orient the eelgrass for display purposes so that it looks more realistic
    set age ticks          ;; Remember what month this eelgrass was planted
  ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;; EELGRASS MISC ;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
to graze-eelgrass
  ;; algorithm representing grazing by waterfowl and other consumers.  This is not yet fully implemented.
  ; little grazing activity in the Chesapeake from NOV-MAR when water temps are <10C. (Wetzel, 1986)
  ; grazing populations decline mid-summer as predatory fish increase. (Wetzel, 1986)
  ; Green Turtle grazing handled via the eat-eelgrass subroutine
end

to stress-eelgrass
  ;;algorithm representing environmental stress on eelgrass from water quality, waves, etc.
  ; at temps above 20C respiration increases at a greater rate than photosynthesis.  mortality above 25C.
  ; epophytic growth blocks sunlight
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;; SEA TURTLE BEHAVIOR ;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to move  ;; turtle procedure
  if count eelgrasses > 0 [                           ; If there are eelgrasses
    face min-one-of eelgrasses [distance myself]      ; turn towards the closes eelgrass
    fd 1                         ; swim 1 meter
  ]
end

to eat-eelgrass  ;; herbivious animal procedure
  ;In this procedure, the animal eats a piece of eelgrass in its patch.
  let meal one-of eelgrasses-here                   ; Look for random nearby eelgrass
  if meal != nobody [                               ; If there is some eelgrass here,
    ask meal [                                      ; eat some eelgrass.
      set veg-mass veg-mass - turtles-eat           ; Reduce the eelgrass by what the turtle just ate.
      if veg-mass <= 0 [ die ]                      ; The eelgrass clump dies if the turtle ate all of it.
    ]
    set energy energy + 2                           ; Put that eelgrass in the turtle's belly.
    if energy > 20 [set energy 20]                  ; Don't exceed 20 energy (stomach + fat).
  ]
end

to death  ;; turtle procedure
  ;; when energy dips below zero, die
  if energy < 0 [ die ]
end



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;; SETUP ROUTINES ;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
to setup-depth
  ;Create a slope to the bay floor.  Per Wetzel (1986), tidal variations do not influence
  ;estimated modeling coefficients in the Chesapeake, so depth is estimated as static.
  set depth (0 - pycor) / max-pycor * 0.75 + 1.25  ;Set depth at .5m (top of window) slowing linearally to 2.0m (bottom of window)
  set pcolor scale-color cyan depth 2.5 -1         ;Display deeper depths as a darker shade of cyan
end

to Display-month-year
  ;; Display the month, year, and temperature (C) in the upper-right corner
  ask patch (max-pxcor - 1) (max-pycor - 3) [
    set plabel array:item month-name (month - 1)
    set plabel word word word word word plabel " " (floor ((ticks - 2) / 12) + Planting-Year + 1) ", " round temperature "C"
  ]
end

to plant-initial-eelgrass-pattern
  ;; Eelgrass is planted in the pattern specified by the picklist

  ;; Plant a single clump in the center of the model
  if eelgrass-pattern = "Single clump" [
    set x 0
    set y 0
    plant-eelgrass
  ]

  ;; Plant a spaced 100m square with clumps every 5m
  if eelgrass-pattern = "100m Dotted Square"  or eelgrass-pattern = "100m Dotted Inscribed Square" [
    set x -50
    set y -50
    repeat 21 [
      plant-eelgrass
      set y y + 5
    ]
    set x 50
    set y -50
    repeat 21 [
      plant-eelgrass
      set y y + 5
    ]
    set x -45
    set y -50
    repeat 19 [
      plant-eelgrass
      set x x + 5
    ]
    set x -45
    set y 50
    repeat 19 [
      plant-eelgrass
      set x x + 5
    ]
  ]

  ;; Plant a spaced 100m line with clumps every 5m
  if eelgrass-pattern = "100m Dotted Line" [
    set x 0
    set y -50
    repeat 21 [
      plant-eelgrass
      set y y + 5
    ]
  ]


  ;; Plant a 100m square with clumps every 1m
  if eelgrass-pattern = "100m Square" or eelgrass-pattern = "100m Inscribed Square" [
    set x -50
    set y -50
    repeat 101 [
      plant-eelgrass
      set y y + 1
    ]
    set x 50
    set y -50
    repeat 101 [
      plant-eelgrass
      set y y + 1
    ]
    set x -49
    set y -50
    repeat 99 [
      plant-eelgrass
      set x x + 1
    ]
    set x -49
    set y 50
    repeat 99 [
      plant-eelgrass
      set x x + 1
    ]
  ]

  ;; Plant a 50m circle with clumps every 1m
  if eelgrass-pattern = "50m Circle" or eelgrass-pattern = "100m Inscribed Square" [
    let theta 0
    repeat 157 [                ; A 50m circle has a circumference of 157 and requires 157 clumps
      set x 25 * sin (theta)    ; Convert polar coordinates to cartesian for planting
      set y 25 * cos (theta)    ; Convert polar coordinates to cartesian for planting
      plant-eelgrass
      set theta theta + 2.293   ; At 25m radius, 2.293 degrees is a 1m arclength.
    ]
  ]

  ;; Plant a 50m circle with clumps every 5m
  if eelgrass-pattern = "50m Dotted Circle" or eelgrass-pattern = "100m Dotted Inscribed Square" [
    let theta 0
    repeat 31 [                ; A 50m circle has a circumference of 157 and requires 31 clumps 5.064, apart
      set x 25 * sin (theta)    ; Convert polar coordinates to cartesian for planting
      set y 25 * cos (theta)    ; Convert polar coordinates to cartesian for planting
      plant-eelgrass
      set theta theta + 11.612   ; At 25m radius, 2.293 degrees is a 1m arclength.
    ]
  ]

  ;; Plant a 50m grid with clumps 7m apart  (actually 49m, not 50m)
  if eelgrass-pattern = "50m Checkerboard" [
    set x -24.5
    repeat 7 [
      set y -24.5
      repeat 7 [
        plant-eelgrass
        set y y + 7
      ]
      set x x + 7
    ]
  ]

end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;; FREE PLANT MODE ;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
to free-plant
  ;If the user clicks while in free plant mode, plant a clump where the user clicked
  if mouse-down? [
    set x mouse-xcor
    set y mouse-ycor
    plant-eelgrass
  ]
  ;We tick to render the new seedling so the user can see it, but only if the model hasn't already been started
  if not simulation-started [
    ;Remember how much eelgrass we planted so that we can tell the % increase in growth
    set initial-extent-eelgrass count eelgrasses        ; # of initial seedlings
    set max-extent-eelgrass initial-extent-eelgrass     ; the initial max = initial seedlings

    tick                                                ; refresh the display so the user can see the new seedling
    reset-ticks                                         ; set ticks back to 0
    clear-plot                                          ; clears the output plot
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
327
14
941
649
75
75
4.0
1
14
1
1
1
0
0
0
1
-75
75
-75
75
1
1
1
ticks
30.0

SLIDER
7
190
154
223
migrating-green-turtles
migrating-green-turtles
0
50
0
1
1
NIL
HORIZONTAL

SLIDER
169
191
307
224
turtles-eat
turtles-eat
0.0
4
0.4
.1
1
NIL
HORIZONTAL

SLIDER
171
526
310
559
mortality-calibration
mortality-calibration
0.0
4
0.1
.1
1
NIL
HORIZONTAL

BUTTON
8
28
77
61
setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
90
28
157
61
go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

PLOT
9
297
322
506
populations
time
pop.
0.0
24.0
0.0
100.0
true
true
"" ""
PENS
"Eelgrass Shoot Mass" 1.0 0 -14439633 true "" "plot sum [veg-mass] of eelgrasses"
"Eelgrass Rhizome Mass" 1.0 0 -10402772 true "" "plot sum [rhizome-mass] of eelgrasses"
"Temperature" 1.0 0 -2139308 true "" "plot temperature"
"Epiphyte Mass" 1.0 0 -8330359 true "" "plot .22 * sum [veg-mass] of eelgrasses"

MONITOR
88
244
164
289
Shoot Mass
sum [veg-mass] of eelgrasses
3
1
11

MONITOR
164
244
240
289
Rhizome Mass
sum [rhizome-mass] of eelgrasses
3
1
11

MONITOR
240
244
317
289
Epiphyte Mass
.22 * sum [veg-mass] of eelgrasses
3
1
11

TEXTBOX
7
171
147
190
Sea turtle settings
11
0.0
0

TEXTBOX
9
68
161
86
Simulation settings
11
0.0
0

BUTTON
172
30
291
63
Step one Month
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

CHOOSER
169
80
308
125
eelgrass-pattern
eelgrass-pattern
"Single clump" "100m Square" "100m Dotted Square" "100m Inscribed Square" "100m Dotted Inscribed Square" "100m Dotted Line" "50m Circle" "50m Dotted Circle" "50m Checkerboard" "Free Plant Mode"
6

SLIDER
18
566
158
599
Rzms-calibration
Rzms-calibration
0
20
1
.1
1
NIL
HORIZONTAL

SLIDER
18
606
157
639
Rzmr-calibration
Rzmr-calibration
0
20
3.6
.1
1
NIL
HORIZONTAL

SLIDER
18
526
158
559
Td-calibration
Td-calibration
0
20
4.1
.1
1
NIL
HORIZONTAL

TEXTBOX
100
509
250
527
Model Calibration Settings
11
0.0
1

SLIDER
170
648
308
681
Global-Warming
Global-Warming
-3
5
0
1
1
NIL
HORIZONTAL

SLIDER
171
566
309
599
Max-clumps-per-patch
Max-clumps-per-patch
0
100
10
1
1
NIL
HORIZONTAL

SLIDER
171
608
308
641
Rhizome-shoot-distance
Rhizome-shoot-distance
0
1
0.5
.05
1
NIL
HORIZONTAL

SLIDER
7
129
154
162
Months-of-experiment
Months-of-experiment
0
120
72
3
1
NIL
HORIZONTAL

SLIDER
7
90
154
123
Planting-Year
Planting-Year
1997
2020
2001
1
1
NIL
HORIZONTAL

MONITOR
13
244
88
289
% Growth
(max-extent-eelgrass / initial-extent-eelgrass) * 100 - 100
0
1
11

SLIDER
19
646
157
679
seed-survival
seed-survival
0
.20
0.1
.01
1
NIL
HORIZONTAL

BUTTON
175
131
296
164
Free Plant Mode
Free-plant
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

@#$#@#$#@
## WHAT IS IT?

This model explores Zostera marina L. (eelgrass) restoration efforts by modeling the seasonal growth and spread of eelgrass in a shallow coastal area in the Chesapeake Bay or adjacent Delmarva Coast.

## HOW IT WORKS

This model simulates growth and reproduction of newly planted eelgrass.  Eelgrass is planted (the initial pattern is shown visually as yellow eelgrass clumps) and season growth and dieback is simulated for a number of years with each tick representing one month of time.

Eelgrass growth and reproduction are modeled using adapted mathmatical metabolic growth models published by the Virginia Institute of Marine Science researchers (Jarvis et al, 2014; Buzzelli et al, 1999; Wetzel & Neckles, 1986).  Eelgrass will continue to grow and reproduce as long as environmental conditions permit and as long as it is not eaten by grazing animals.

The environment is a tidal area with a gently sloping sand bottom from 0.5m deep (top of the screen) to 2m deep (bottom of the screen) and the incoming tide rising to the right.  Water temperatures are based upon long-term average temperatures for the region.

This model begins by planting a number of patches of eelgrass in a set pattern similar to those used 1999-2008 restoration efforts.  The eelgrasses germinate in early winter, grow through the spring and early summer, and die off in the late summer.  Mature healthy plants spread by rhizome growth a random short distance and direction.  Mature healthy plants more than one year old release seeds in the late summer, most of which are carried by the rising tide and some of which are randomly distributed.

Green Turtles are modeled as grazing predators, migrating into the Chesapeake Bay area in Spring and migrating south in early Fall.  These turtles will seek out and eat the nearest eelgrass. Each turtle feeds 30 times per month.  If a turtle does not find enough grass to eat, it will die (leave).


## HOW TO USE IT

1. Select one of the seed patterns from the drop-down list.
2. Use the Planting-Year slider to select which year to begin the simulation
3. Use the months-of-experiment slider to select the number of months to run the model
4. Select whether to keep water temperatures nominal (0C) or to simulate global warming by increasing water tempreature a number of degrees Celcius.
5. Select the number of green turtles which will migrage each year.
6. Select how much eelgrass each turtle will eat.
7. If desired, adjust the model calibration sliders.  Default selections were found from previous modeling to yield a model that closely resembled real-world experimental results.

9. Press Setup to reset the model and plant the seeds (seed planting locations shown in yellow)
10. If desired, use free-plant mode by depressing the free plant mode button and manually drawing eelgrass clumps using your mouse.  This may be done even if another pattern has been selected.
11. Press the Go button to run the simulation. Alternately, you may press the Step One Month button to go one month at a time.

EELGRASS-PATTERN: the pattern in which eelgrass seeds will be planted
PLANTING-YEAR: seeds will be modeled as germinating in November of this year
MONTHS-OF-EXPERIMENT: selectable in 12 month increments, the model will be run this number of months
GLOBAL-WARMING: water temperature, in degrees celcius, that will be added to the typical average bay water temperature
MIGRATING-GREEN-TURTLES: the number of green turtles that migrate in each year.
TURTLES-EAT: how much eegrass shoot biomass is eaten each day by each green turtle
TD-CALIBRATION: calibration coefficient for the amount of biomass translocated down from shoots to rhizomes.
RZMS-CALIBRATION: calibration coefficient for shoot respiration
RZMR-CALIBRATION: calibration coefficient for rhizome respiration
SEED-SURVIVAL: calibration coefficient for what fraction of seeds survive
MORTALITY-CALIBRATION: calibration coefficient for what fraction of shoots randomly die
MAX-CLUMPS-PER-PATCH: calibration coefficient for limiting density by limiting the number of eelgrass clumps permitted in each 1x1m patch.
RHIZOME-SHOOT-DISTANCE: calibration coefficient for how far new shoots are created by rhizome spread

## THINGS TO NOTICE

Do you see the exponential growth?  Do you see the difference between newer and older beds after several years of growth, or do they appear the same?

Do you notice the difference in growth between different seed patterns for different amounts of time?  Are there similarities?  What are the differences?


## THINGS TO TRY

Try adjusting the number of sea turtles to see the impact on smaller beds.  Does the impact lessen as the bed grows?  Is there a significant difference when many turtles are added or if they're hungrier?

Try using free plant mode to see how your pattern grows.

Try using BehaviorSpace to compare the results of different models.


## EXTENDING THE MODEL

This model only included the basic metabolic model and does not include many real-world limitations such as nitrogen reduction, CO2 reduction, and turbidity.  Implementation of the full model developed by Jarvis et al (2014) is anticipated to yield much higher quality results.

Can you add other predators and stressors such as fowl, crownose rays, and crabs?

Can you add stressful events, like hurricanes?

## VERSION CHANGES
2.10:
-Added dotted versions of the squares and circle with 5m spacing between clumps

2.9:
-Added free plant mode
-Added additional code comments and documentation
-Added error trapping for when green turtles eat all the eelgrass
-Corrected conservation of rhizome mass during asexual reproduction

2.8:
-Added additional code comments

2.7:
-Added seed broadcast to model

2.6:
-Adjusted dates to 2002-2008 to match Orth (2009) restoration records of the DelMarVa
-Adjusted size to match actual Chesapeake planting patterns.  Added 50m circle.
-Changed sea turtles to migration pattern

2.5:
-Removed Epiphyte code
-Added factor into rhysome respiration to replicate real-life winter reduction of biomass
-Added depth impact to PAR
-Added rhizome reproduction back in

2.0 - 2.4:
-Incremental model development

2.0:
-Changed patch-based eelgrass sections to turtle-based eelgrass clumps.

1.0:
-Early developmental model

## REFERENCES AND CREDITS

This model was created in NetLogo by Stephen Fehr, Old Dominion University, June 2016.

Thank you to Nima Shahriari and Adrian Gheorghe of Old Dominion University for the insights and instruction used to develop this model and to Uri Wilensky for developing Netlogo.  Special thanks to the Virginia Institute of Marine Science team and all involved in Chesapeake Bay SAV restoration.

Buzzelli, C., Wetzel, R., Meyers, M. (1999). A Linked Physical and Biological Framework to Assess Biogeochemical Dynamics in a Shallow Estuarine Ecosystem.  Estuarine, Coastal and Shelf Science 49.

Jarvis, J., Brush, M., Moore, K. (2014).  Modeling loss and recovery of Zostera marina beds in the Chesapeake Bay: The role of seedlinks and seed-bank viability.  Aquatic Botany 113.

Orth, R., et al (2009) Restoration of Seagrasses in Virginia Seaside Bays Year 6 (Oct 1, 2007, to Dec 31, 2008) and Summary FY 2007 Task 10.01.  doi: 10.1.1.570.1755

Railsback, S., Grimm, V. (2012). Agent-Based and Individual-Based Modeling: A Practical Introduction.  Princeton University Press: Princeton, NJ.

Wetzel, R., Neckles, H. (1986).  A Model of Zostera Marina Photosynthesis and Growth: Simulated Effects of Selected Physical-Chemical Variables and Biological Interactions.  Aquatic Botany, 26.

Wilensky, U. (1999). NetLogo. http://ccl.northwestern.edu/netlogo/. Center for Connected Learning and Computer-Based Modeling, Northwestern University. Evanston, IL.

<!-- 1997 2000 -->
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

bird side
false
0
Polygon -7500403 true true 0 120 45 90 75 90 105 120 150 120 240 135 285 120 285 135 300 150 240 150 195 165 255 195 210 195 150 210 90 195 60 180 45 135
Circle -16777216 true false 38 98 14

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

eelgrass
true
0
Line -7500403 true 120 45 105 285
Line -7500403 true 180 30 195 255
Line -7500403 true 75 45 105 285
Line -7500403 true 225 75 195 255
Line -7500403 true 15 105 30 210
Line -7500403 true 285 120 255 255
Line -7500403 true 150 225 150 300

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.3.1
@#$#@#$#@
set grass? true
setup
repeat 75 [ go ]
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="Seed Pattern Yields" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="72"/>
    <metric>(max-extent-eelgrass / initial-extent-eelgrass) * 100 - 100</metric>
    <metric>count eelgrasses</metric>
    <metric>sum [veg-mass] of eelgrasses</metric>
    <metric>sum [rhizome-mass] of eelgrasses</metric>
    <enumeratedValueSet variable="mortality-calibration">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Months-of-experiment">
      <value value="72"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="eelgrass-pattern">
      <value value="&quot;100m Square&quot;"/>
      <value value="&quot;100m Inscribed Square&quot;"/>
      <value value="&quot;50m Circle&quot;"/>
      <value value="&quot;50m Checkerboard&quot;"/>
      <value value="&quot;Single clump&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Td-calibration">
      <value value="4.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Rzms-calibration">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Global-Warming">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Planting-Year">
      <value value="2003"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max-clumps-per-patch">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migrating-green-turtles">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Rzmr-calibration">
      <value value="3.3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seed-survival">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="turtles-eat">
      <value value="0.4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Rhizome-shoot-distance">
      <value value="0.5"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
