;;____________________________
;
;;   SETUP AND HOUSEKEEPING
;;____________________________

breed [ parties party ]

globals [
  total-votes
  max-voteshare            ; largest vote share on any patch
  mean-voterx              ; mean voter x-coordinate
  mean-votery              ; mean voter y-coordinate
  
  cycle                    ; cycle (or campaign) number or tick
  election                 ; election number or tick

  mean-eccentricity        ; mean Euclidean distance of parties from (mean-voterx, mean-votery)
  voter-misery             ; mean quadratic Euclidean voter distance from closest party
  enp                      ; effective number of parties = 1/(sum of squared party vote shares)
  
  rule-number              ; rule number
  rule-list                ; list of available decision rules
  rule-voteshare           ; list of vote shares won by the set of parties using each rule
  rule-pcount              ; number of parties using rule
  rule-eccent              ; mean eccentricity of parties using rule
]

parties-own [
  rule                      ; party's parameterized decision rule
  
                     ; decision rule parameters
  speed                     ; distance each party moves per tick
  
                     ; indicators
  mysize                    ; current party size
  old-size                  ; party's size previous tick
  old-x                     ; x-coordinate of my position at previous election
  old-y                     ; y-coordinate of my position at previous election
  age                       ; number of elections survived since birth
  fitness                   ; party's evolutionary fitness
  eccentricity              ; party's Euclidean distance from (mean-voterx, mean-votery)
]

patches-own [
  votes                     ; number of voters on patch
  vote-share                ; proportion of total voters on patch
  closest-party             ; party with smallest Euclidean distance from patch
  misery                    ; quadratic distance from closest party, weighted by patch's vote share
]

to setup
  clear-all
  file-close  
    
  if (endogenous-birth = false) [set misery-alpha 0 set misery-beta 0] 
    ; if the endogenous-birth option has not been chosen, set misery-alpha and misery-beta to 0 
  
  if (birth-death-file = true) [
    if (file-exists? bd-file-name) [file-delete bd-file-name]
    file-open bd-file-name
    ; if a birth-death file has been requested, delete any previous birth-death files with name equal to bd-file-name and
    ; open a new file with name equal to bd-file-name. 
  ]

  set rule-list (list "sticker" "aggregator" "hunter")
  set rule-number n-values length(rule-list) [?]
  set rule-voteshare n-values length(rule-list) [0]
  set rule-pcount n-values length(rule-list) [0]
  set rule-eccent n-values length(rule-list) [-1]
    ; initialize rule measure vectors: rule-list is the vector of all of rules examined; rule-number is a unique number for each
    ; rule; rule-voteshare and rule-pcount are initialized at 0 for each rule; and rule-eccent is initialized at -1 for each rule
    
  create-parties 1
  ask parties [set fitness 1 set heading random-float 360 jump random-float 30 set old-x xcor set old-y ycor  
               set age 0 set size 2 random-pick color-myself ]          
               ;; every run starts with a single party, which has a random position and rule picked from the rule list
          
  ask patches [ 
      let x1 (pxcor - x-mean1) / sd-1
      let y1 (pycor - y-mean1) / sd-1      
      set votes votes1 * exp (-0.5 * ( x1 ^ 2 + y1 ^ 2)) / (2 * pi * sd-1 ^ 2)
        ;; votes1, x_mean1, y_mean1, sd_1 = votes, mean and standard deviation of subpopulation 1, read from interface
        ;; each patch's votes arising from subpopulation 1 =  votes1 * bivariate normal density with mean1, sd_1, rho = 0
          
      let x2 (pxcor - x-mean2) / sd-2
      let y2 (pycor - y-mean2) / sd-2      
      set votes votes + votes2 * exp (-0.5 * ( x2 ^ 2 + y2 ^ 2)) / (2 * pi * sd-2 ^ 2)
        ;; add votes to each patch from subpopulation 2, calculated as above
          
      let x3 (pxcor - x-mean3) / sd-3
      let y3 (pycor - y-mean3) / sd-3      
      set votes votes + votes3 * exp (-0.5 * ( x3 ^ 2 + y3 ^ 2)) / (2 * pi * sd-3 ^ 2)
        ;; add votes to each patch from subpopulation 3, calculated as above
      ]
      
  set total-votes sum [ votes ] of patches
  type "Total votes at all locations = " type round(total-votes)     
        ;; add total of votes on all patches and output this to the command window
  
  ask patches [set vote-share votes / total-votes]      
      ;calculate each patch's vote share
  
  set mean-voterx sum [ pxcor * vote-share ] of patches
  set mean-votery sum [ pycor * vote-share ] of patches      
  type "   Mean voter x = " type round(mean-voterx) 
  type "   Mean voter y = " print round(mean-votery)
      ;; calculate center (mean) of voter distribution on each dimension as sum of (patch positions weighted by vote share)
      ;; output this to the command window 
  
  set max-voteshare max [ vote-share ] of patches
  ask patches [set pcolor scale-color red vote-share 0 max-voteshare ] 
      ;; color patches red with density proportional to vote shares
  
  update-support
      ;; ask voters to choose closest party and calculate relative success of different rules
  update-rule-measures
      ;; update rule-measure vectors
  
end  

; ******* parameter setup buttons
to random-pop
   set sd-1 5 set sd-2 5 set y-mean1 0 set y-mean2 0
   set x-mean2 precision (random-float 15)  2    set x-mean1 0 - x-mean2              
   set votes1 500000 + random 166667  set votes2 1000000 - votes1
end

to symmetric-pop
   set sd-1 10 set x-mean1 0 set y-mean1 0 set votes1 1000000  set votes2 0 set votes3 0 set x-mean2 0
end


;;____________________________
;
;;   PARTY DYNAMICS
;;____________________________

to stick
      ;; do nothing
end 


to aggregate
   if (mysize > 0) 
   [
     set xcor (sum [votes * pxcor] of patches with [closest-party = myself] / mysize)
     set ycor (sum [votes * pycor] of patches with [closest-party = myself] / mysize)
   ]
      ;; set party x, y positions at the mean x, y positions of party members; maintain current position if zero supporters
end  


to hunt
  ifelse (mysize > old-size) [jump speed] [set heading heading + 90 + random-float 180  jump speed]
      ;; hunter makes a move of size speed in same direction as previous move if this increased party support
      ;; else reverses direction and makes a move of size speed in on a heading chosen from the 180 degree arc now faced 
  set old-size mysize 
      ;; remember party size for next cycle
end


;;____________________________
;;
;;   MAIN CONTROL SUBROUTINES
;;____________________________

to update-support
  ask patches [set closest-party min-one-of parties [distance myself]]
      ;; patches find their closest party
  ask parties [set mysize sum [votes] of patches with [closest-party = myself]] 
      ;; each party sums the votes on patches for which it is the closest party
end

to calculate-election-results
  set election election + 1
  update-party-measures
  update-rule-measures
  measure-enp
  measure-eccentricity
  measure-misery
  party-death
  party-birth
end
  ; execute the above set of commands at every election tick

to update-party-measures                                   
  ask parties [
      set fitness fitness-alpha * fitness + (1 - fitness-alpha) * mysize / total-votes
                 ;; parties recursively update their fitness as: (alpha)*(previous fitness) + (1-alpha)*(current vote share) 
      set age age + 1 set old-x xcor set old-y ycor
  ]
end

to update-rule-measures
   (foreach rule-number rule-list [ 
      set rule-voteshare replace-item ?1 rule-voteshare sum [mysize / total-votes] of parties with [rule = ?2]
          ;; calculate the current support level of all parties using each rule
        
      set rule-pcount replace-item ?1 rule-pcount count parties with [rule = ?2]
          ;; count the number of parties using each rule

      ifelse (sum [mysize] of parties with [rule = ?2] > 0) 
        [
        set rule-eccent replace-item ?1 rule-eccent mean [eccentricity] of parties with [rule = ?2] 
        ]
          ;;calculate the mean of eccentricity, policy loss and policy shift of all parties using each rule
          
        [ 
        set rule-eccent replace-item ?1 rule-eccent -1 
        ]
          ;;these measures have no meaning when no party uses a rule
    ])  
end

to measure-enp
  set enp (total-votes ^ 2) / (sum [mysize ^ 2] of parties)
     ;; calculate the enp of the system
end

to measure-eccentricity
  ask parties [set eccentricity sqrt ((xcor - mean-voterx) ^ 2 + (ycor - mean-votery) ^ 2) / 10] 
     ;; calculate each party's eccentricity, its Euclidean distance from the center of the voter distribution
  set mean-eccentricity mean [eccentricity] of parties
     ;; calculate the mean eccentricity of all parties in the system
end

to measure-misery
   ask patches [set misery misery-alpha * misery + (1 - misery-alpha) * ((distance closest-party ^ 2) / 100) * vote-share]
   set voter-misery sum [misery] of patches
      ;; patch misery is misery at t-1, updated by mean quadratic Euclidean distance of patch from closest party, 
      ;; weighted by patch vote share
      ;; mean voter "misery" is thus updated mean quadratic Euclidean distance of each voter from his/her closest party
end

to party-death
   ask parties [if (fitness < survival-threshold and count parties > 2) 
       [
          if (birth-death-file = true) [
              file-write votes1 file-write x-mean1 file-write votes2 file-write x-mean2 file-write fitness-alpha file-write survival-threshold
              file-write campaign-ticks file-write misery-alpha file-write misery-beta
              file-write election file-write "death" file-write rule file-write who 
              file-write precision xcor 4 file-write precision ycor 4 file-write "age" file-write age file-print ""
          ] 
          die 
          ask patches [set closest-party min-one-of parties [distance myself]]
       ]]   
                 ;; parties whose updated fitness falls below the survival threshold write out their data and die
                 ;; as long as there are at least two parties
end

to party-birth
  ifelse (endogenous-birth = true)
    [ ask one-of patches with [distancexy 0 0 < 30]
      [ if (random-float 1 < (misery-beta * misery * 1000)) [sprout-parties 1 [initialize-party] ]]]
        ;; pick a random patch within three standard deviations of the origin.
        ;; the probability this patch sprouts a new party is proportional to (misery-beta)*(patch misery)
        ;; the greater patch misery, the higher the probability the patch sprouts a new party.
        ;; NB patch misery is scaled, in measure-misery above, to the share of all voters on the patch, 
        ;; this share maxes at 0.00159 for the (0,0) patch in a unimodal (0,10) distribution and is 0.0002 at patch (20,0) in this distibution
        ;; this explains the scaling up of the patch misery score by 1000 and the units of beta are thus sui generis to the simulation
        ;; the greater beta, however, the more sensitive are voters on a patch to a given level of misery.
        ;; new-born parties intially locate at the position of the "sprouting" patch.
    
    [ create-parties 1 [set heading random-float 360 jump random-float 30 initialize-party] ]
        ;; non-endogenous initial party locations take a random walk within 30 from the origin
end

to initialize-party
  ifelse (count parties = 0) [set fitness 1] [set fitness 1 / count parties] 
  set heading random-float 360 set old-x xcor set old-y ycor set speed 1 set age 0 set size 1.5                                                               
    
  random-pick                      
  color-myself
  
  if (birth-death-file = true) [
    file-write votes1 file-write x-mean1 file-write votes2 file-write x-mean2 file-write fitness-alpha file-write survival-threshold
    file-write campaign-ticks file-write misery-alpha file-write misery-beta
    file-write election file-write "birth" file-write rule file-write who 
    file-write precision xcor 4 file-write precision ycor 4  file-print ""
  ]
     ;; initialize a new party by picking a random rule from the rule list and 
     ;; write out your starting data before handing control back to the observer
end

to random-pick
    set rule one-of rule-list
    ;; randomly pick a rule from the list in the setup routine
end

to color-myself
  if (rule = "sticker") [set color yellow]
  if (rule = "aggregator") [set color lime]
  if (rule = "hunter") [set color violet]
end

to adapt
  if (rule = "sticker") [stick]
  if (rule = "aggregator") [aggregate]
  if (rule = "hunter") [hunt]
   ;; NB stickers do nothing
end


;;____________________________
;
;;   MAIN CONTROL ROUTINE
;;____________________________

to go
  repeat campaign-ticks
  ; for each election, repeat the following commands [campaign-ticks] times
  [
    set cycle cycle + 1
    ask parties [adapt]
    update-support
    if (remainder cycle campaign-ticks = 0 and cycle != 0) [calculate-election-results]
     ; increase the cycle number; ask parties to adapt using their rules; update party support;
     ; and if the remainder of the cycle number divided by campaign ticks is zero, calculate election results
    ]
  
end
@#$#@#$#@
GRAPHICS-WINDOW
499
8
1050
580
35
35
7.62
1
10
1
1
1
0
0
0
1
-35
35
-35
35
0
0
1
ticks

BUTTON
13
11
82
53
Setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

SLIDER
42
338
156
371
votes1
votes1
0
1000000
1000000
1000
1
NIL
HORIZONTAL

SLIDER
154
338
255
371
x-mean1
x-mean1
-40
40
-15
1
1
NIL
HORIZONTAL

SLIDER
253
338
354
371
y-mean1
y-mean1
-40
40
0
1
1
NIL
HORIZONTAL

SLIDER
352
338
452
371
sd-1
sd-1
0
40
10
0.5
1
NIL
HORIZONTAL

SLIDER
42
370
155
403
votes2
votes2
0
1000000
1000000
1000
1
NIL
HORIZONTAL

SLIDER
154
370
254
403
x-mean2
x-mean2
-40
40
15
1
1
NIL
HORIZONTAL

SLIDER
254
370
353
403
y-mean2
y-mean2
-40
40
0
1
1
NIL
HORIZONTAL

SLIDER
352
370
452
403
sd-2
sd-2
0
40
10
0.5
1
NIL
HORIZONTAL

SLIDER
42
401
155
434
votes3
votes3
0
1000000
0
1000
1
NIL
HORIZONTAL

SLIDER
154
401
254
434
x-mean3
x-mean3
-40
40
0
1
1
NIL
HORIZONTAL

SLIDER
253
401
353
434
y-mean3
y-mean3
-40
40
0
1
1
NIL
HORIZONTAL

SLIDER
352
401
452
434
sd-3
sd-3
0
40
1
0.5
1
NIL
HORIZONTAL

TEXTBOX
119
442
407
461
Population designer: Equilateral = (0 17)(-15 -9)(15 -9)
11
0.0
0

BUTTON
82
11
150
53
Go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL

MONITOR
219
262
294
307
Aggregator
item 1 rule-voteshare
2
1
11

MONITOR
294
262
369
307
Hunter
item 2 rule-voteshare
2
1
11

MONITOR
307
52
376
97
Eccentricity
mean-eccentricity
2
1
11

MONITOR
239
52
307
97
Misery
voter-misery
2
1
11

TEXTBOX
190
310
354
330
Vote share after last election\\n
11
0.0
1

SLIDER
194
124
338
157
fitness-alpha
fitness-alpha
0
.95
0.5
.01
1
NIL
HORIZONTAL

SLIDER
194
156
338
189
survival-threshold
survival-threshold
0
1.0
0.1
.01
1
NIL
HORIZONTAL

SLIDER
194
189
338
222
campaign-ticks
campaign-ticks
1
50
20
1
1
NIL
HORIZONTAL

MONITOR
144
262
219
307
Sticker
item 0 rule-voteshare
2
1
11

MONITOR
375
52
444
97
ENP
enp
2
1
11

TEXTBOX
264
29
420
47
Measures from last election
11
0.0
1

MONITOR
172
52
240
97
N parties
count parties
0
1
11

MONITOR
51
52
120
97
Election
election
1
1
11

SWITCH
40
218
175
251
birth-death-file
birth-death-file
1
1
-1000

SLIDER
40
125
175
158
misery-alpha
misery-alpha
0
1
0
.01
1
NIL
HORIZONTAL

SLIDER
40
154
175
187
misery-beta
misery-beta
0
1
0
.01
1
NIL
HORIZONTAL

SWITCH
40
186
175
219
endogenous-birth
endogenous-birth
0
1
-1000

TEXTBOX
232
108
358
126
Environment
11
0.0
1

TEXTBOX
45
109
165
127
Party birth parameters
11
0.0
1

TEXTBOX
131
525
430
553
Stickers yellow; Aggregators green; Hunters violet
11
0.0
1

BUTTON
205
464
309
497
Random population
random-pop
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

INPUTBOX
365
124
478
184
bd-file-name
manual_bd.txt
1
0
String

BUTTON
384
219
464
252
file-close
file-close
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

TEXTBOX
357
187
494
229
Hit file-close after a manual \\nrun with a birth-death file
11
0.0
1

@#$#@#$#@
WHAT IS IT?
-----------
This implements the model of party competition, with endogenous party birth and death, specified in Chapter 6 of Michael Laver and Ernest Sergenti's book, Party competition: an agent based model (Princeton University Press, 2012). A full description and analysis of the model can be found in this book and is not repeated here.

Party positions and voter ideal points are defined in two dimensions of policy/ideology, directly analogous to the dimensions used in other "spatial" models of political competition.  The horizontal dimension might, for example, be seen as describing left-right economic policy positions; the vertical dimension might be a liberal-conservative policy dimension on matters such as abortion, sexuality, euthanasia. 

VOTERS always vote for the closest party. The preference structure of the voting population can be designed as if this is an aggregate of up to three subpopulations (though only two are investigated by Laver and Sergenti and the default setting on the interface sets the size of the third subpopulation at zero). Voters in each subpopulation have normally distributed ideal points, and each subpopulation is characterized by: the number of voters it comprises; the standard deviation of the distribution of its voters' ideal points, and the means of this distribution on the x and y dimensions. All of these parameters can be set using the sliders in the _Population Designer_ panel near the bottom of the interface. Alternatively, the _random population_ button picks these at random.

PARTY LEADERS compete with each other by offering policies to potential supporters. They use one of three _species_ of decision rule _ Sticker, Aggregator, Hunter _ to select a party policy position. These riles are specified in the _Party dynamics_ section of the code.

DYNAMICS OF PARTY COMPETITION. The baseline dynamics of the model iterate forever.  (1) Voters support their closest party.  (2) Given profile of voter support for parties, leaders adapt party policy positions using their decision rule. (3) Go to 1.

The set of surviving political parties is fully endogenous to the model. 

EXISTING PARTIES DIE if their updated fitness, denominated in vote share, falls below a system survival threshold. The party survival threshold, and the memory parameter in the fitness updating regime, can be set using the sliders in the _Environment_ panel near the top of the interface.  

NEW PARTIES ARE BORN at the ideal points of disgruntled voters. Parameters of the party birth regime can be set using the sliders in the _Party birth_ panel near the top of the interface.  There is a switch on the interface that turns off endogenous party birth and instead randomly generates new party births at random locations. (This was not investigated systematically by Laver and Sergenti.)

Model ticks are divided into CAMPAIGN TICKS and ELECTION TICKS. Party leaders adapt their positions during campaign ticks but receive no rewards or punishments. Parties can only die or be born on election ticks. The number of campaign ticks per election tick can be set using the slider in the _Environment_ panel near the top of the interface.  

HOW TO USE IT
-------------
SETUP sets up parties, supporters and system parameters as specified above.  GO starts and stops the simulation using current parameters.
(Hitting SETUP while GO is still pressed very occasionally causes an error depending where precisely the program is when setup is hit; this easily retrieved by unpressing GO and pressing SETUP again.)

RUNNING EXPERIMENTS.  Laver and Sergenti designed a large computational experiment, and report results of this, in Chapter 6 of their book. Although the _production_ run was executed on a high performance cluster, precisely equivalent smaller scale experiments can easily be run using Behavior Space. Sketch runs for all results reported in Laver and Sergenti were generated using Behavior Space on a normal laptop.

DATA OUTPUT. Standard data output is via Behavior Space experiments used in the normal way. There is a separate data channel that writes out information on party births and deaths only when these occur.  This is activated by a switch and a file name on the interface.


WHAT TO PLAY WITH
-----------
Laver and Sergenti report results from a carefully controlled computational experiment and only investigate electorate with two subpopulations. There are infinitely many alternative populations for you to explore using the population designer. There are also many parameterizations of the competitive environment, and the party birth regime, not explored by Laver and Sergenti. You may, for example, want to specify a parameterization of the model you feel corresponds to some real political system that interests you.

By far the most exciting and callenging way forward is to specify and program your own decision rule for party leaders.  Just drop in your coded new rule as a procedure in the party dynamics section, add its name to the rule list, edit it in to the _adapt_ and _color-myself_ procedures, and add a reporter for your rule_s vote share to the interface. You_re good to go!

CREDITS AND REFERENCES
----------------------
Programmed by:
 
Michael Laver, Department of Politics, New York University
ml127@nyu.edu

Ernest Sergenti, The World Bank
esergenti@gmail.com
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

link
true
0
Line -7500403 true 150 0 150 300

link direction
true
0
Line -7500403 true 150 150 30 225
Line -7500403 true 150 150 270 225

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 4.1.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="SAH_test" repetitions="1" runMetricsEveryStep="true">
    <setup>random-seed 12345
set votes1 500000 
set x-mean1 7
set y-mean1 0
set sd-1 6.5
set votes2 500000
set x-mean2 -7
set y-mean2 0
set sd-2 6.5
set votes3 0
set x-mean3 0
set y-mean3 0
set sd-3 1.0
set fitness-alpha .5
set survival-threshold .1
set campaign-ticks 20
setup</setup>
    <go>go</go>
    <final>file-close</final>
    <timeLimit steps="10"/>
    <metric>votes1</metric>
    <metric>x-mean1</metric>
    <metric>y-mean1</metric>
    <metric>sd-1</metric>
    <metric>votes2</metric>
    <metric>x-mean2</metric>
    <metric>y-mean2</metric>
    <metric>sd-2</metric>
    <metric>votes3</metric>
    <metric>x-mean3</metric>
    <metric>y-mean3</metric>
    <metric>sd-3</metric>
    <metric>fitness-alpha</metric>
    <metric>survival-threshold</metric>
    <metric>campaign-ticks</metric>
    <metric>election</metric>
    <metric>voter-misery</metric>
    <metric>mean-eccentricity</metric>
    <metric>enp</metric>
    <metric>(item 0 rule-pcount)</metric>
    <metric>(item 0 rule-eccent)</metric>
    <metric>(item 0 rule-voteshare)</metric>
    <metric>(item 1 rule-pcount)</metric>
    <metric>(item 1 rule-eccent)</metric>
    <metric>(item 1 rule-voteshare)</metric>
    <metric>(item 2 rule-pcount)</metric>
    <metric>(item 2 rule-eccent)</metric>
    <metric>(item 2 rule-voteshare)</metric>
  </experiment>
  <experiment name="SAH_7_equal" repetitions="1" runMetricsEveryStep="true">
    <setup>set votes1 500000 
set x-mean1 7
set y-mean1 0
set sd-1 6.5
set votes2 500000
set x-mean2 -7
set y-mean2 0
set sd-2 6.5
set votes3 0
set x-mean3 0
set y-mean3 0
set sd-3 1.0
set comfort-threshold 0
set fitness-alpha .5
set survival-threshold .3
set campaign-ticks 20
set new-party-rule "random-pick"
setup</setup>
    <go>go</go>
    <final>file-close</final>
    <timeLimit steps="20000"/>
    <metric>votes1</metric>
    <metric>x-mean1</metric>
    <metric>y-mean1</metric>
    <metric>sd-1</metric>
    <metric>votes2</metric>
    <metric>x-mean2</metric>
    <metric>y-mean2</metric>
    <metric>sd-2</metric>
    <metric>votes3</metric>
    <metric>x-mean3</metric>
    <metric>y-mean3</metric>
    <metric>sd-3</metric>
    <metric>comfort-threshold</metric>
    <metric>fitness-alpha</metric>
    <metric>survival-threshold</metric>
    <metric>campaign-ticks</metric>
    <metric>election</metric>
    <metric>voter-misery</metric>
    <metric>mean-eccentricity</metric>
    <metric>enp</metric>
    <metric>(item 0 rule-pcount)</metric>
    <metric>(item 0 rule-eccent)</metric>
    <metric>(item 0 rule-votes) / total-votes</metric>
    <metric>(item 1 rule-pcount)</metric>
    <metric>(item 1 rule-eccent)</metric>
    <metric>(item 1 rule-votes) / total-votes</metric>
    <metric>(item 2 rule-pcount)</metric>
    <metric>(item 2 rule-eccent)</metric>
    <metric>(item 2 rule-votes) / total-votes</metric>
  </experiment>
  <experiment name="SAH_8_equal" repetitions="1" runMetricsEveryStep="true">
    <setup>set votes1 500000 
set x-mean1 8
set y-mean1 0
set sd-1 6
set votes2 500000
set x-mean2 -8
set y-mean2 0
set sd-2 6
set votes3 0
set x-mean3 0
set y-mean3 0
set sd-3 1.0
set comfort-threshold 0
set fitness-alpha .5
set survival-threshold .3
set campaign-ticks 20
set new-party-rule "random-pick"
setup</setup>
    <go>go</go>
    <final>file-close</final>
    <timeLimit steps="20000"/>
    <metric>votes1</metric>
    <metric>x-mean1</metric>
    <metric>y-mean1</metric>
    <metric>sd-1</metric>
    <metric>votes2</metric>
    <metric>x-mean2</metric>
    <metric>y-mean2</metric>
    <metric>sd-2</metric>
    <metric>votes3</metric>
    <metric>x-mean3</metric>
    <metric>y-mean3</metric>
    <metric>sd-3</metric>
    <metric>comfort-threshold</metric>
    <metric>fitness-alpha</metric>
    <metric>survival-threshold</metric>
    <metric>campaign-ticks</metric>
    <metric>election</metric>
    <metric>voter-misery</metric>
    <metric>mean-eccentricity</metric>
    <metric>enp</metric>
    <metric>(item 0 rule-pcount)</metric>
    <metric>(item 0 rule-eccent)</metric>
    <metric>(item 0 rule-votes) / total-votes</metric>
    <metric>(item 1 rule-pcount)</metric>
    <metric>(item 1 rule-eccent)</metric>
    <metric>(item 1 rule-votes) / total-votes</metric>
    <metric>(item 2 rule-pcount)</metric>
    <metric>(item 2 rule-eccent)</metric>
    <metric>(item 2 rule-votes) / total-votes</metric>
  </experiment>
  <experiment name="SAH_9_equal" repetitions="1" runMetricsEveryStep="true">
    <setup>set votes1 500000 
set x-mean1 9
set y-mean1 0
set sd-1 5.5
set votes2 500000
set x-mean2 -9
set y-mean2 0
set sd-2 5.5
set votes3 0
set x-mean3 0
set y-mean3 0
set sd-3 1.0
set comfort-threshold 0
set fitness-alpha .5
set survival-threshold .3
set campaign-ticks 20
set new-party-rule "random-pick"
setup</setup>
    <go>go</go>
    <final>file-close</final>
    <timeLimit steps="20000"/>
    <metric>votes1</metric>
    <metric>x-mean1</metric>
    <metric>y-mean1</metric>
    <metric>sd-1</metric>
    <metric>votes2</metric>
    <metric>x-mean2</metric>
    <metric>y-mean2</metric>
    <metric>sd-2</metric>
    <metric>votes3</metric>
    <metric>x-mean3</metric>
    <metric>y-mean3</metric>
    <metric>sd-3</metric>
    <metric>comfort-threshold</metric>
    <metric>fitness-alpha</metric>
    <metric>survival-threshold</metric>
    <metric>campaign-ticks</metric>
    <metric>election</metric>
    <metric>voter-misery</metric>
    <metric>mean-eccentricity</metric>
    <metric>enp</metric>
    <metric>(item 0 rule-pcount)</metric>
    <metric>(item 0 rule-eccent)</metric>
    <metric>(item 0 rule-votes) / total-votes</metric>
    <metric>(item 1 rule-pcount)</metric>
    <metric>(item 1 rule-eccent)</metric>
    <metric>(item 1 rule-votes) / total-votes</metric>
    <metric>(item 2 rule-pcount)</metric>
    <metric>(item 2 rule-eccent)</metric>
    <metric>(item 2 rule-votes) / total-votes</metric>
  </experiment>
  <experiment name="SAH_10_equal" repetitions="1" runMetricsEveryStep="true">
    <setup>set votes1 500000 
set x-mean1 10
set y-mean1 0
set sd-1 5
set votes2 500000
set x-mean2 -10
set y-mean2 0
set sd-2 5
set votes3 0
set x-mean3 0
set y-mean3 0
set sd-3 1.0
set comfort-threshold 0
set fitness-alpha .5
set survival-threshold .3
set campaign-ticks 20
set new-party-rule "random-pick"
setup</setup>
    <go>go</go>
    <final>file-close</final>
    <timeLimit steps="20000"/>
    <metric>votes1</metric>
    <metric>x-mean1</metric>
    <metric>y-mean1</metric>
    <metric>sd-1</metric>
    <metric>votes2</metric>
    <metric>x-mean2</metric>
    <metric>y-mean2</metric>
    <metric>sd-2</metric>
    <metric>votes3</metric>
    <metric>x-mean3</metric>
    <metric>y-mean3</metric>
    <metric>sd-3</metric>
    <metric>comfort-threshold</metric>
    <metric>fitness-alpha</metric>
    <metric>survival-threshold</metric>
    <metric>campaign-ticks</metric>
    <metric>election</metric>
    <metric>voter-misery</metric>
    <metric>mean-eccentricity</metric>
    <metric>enp</metric>
    <metric>(item 0 rule-pcount)</metric>
    <metric>(item 0 rule-eccent)</metric>
    <metric>(item 0 rule-votes) / total-votes</metric>
    <metric>(item 1 rule-pcount)</metric>
    <metric>(item 1 rule-eccent)</metric>
    <metric>(item 1 rule-votes) / total-votes</metric>
    <metric>(item 2 rule-pcount)</metric>
    <metric>(item 2 rule-eccent)</metric>
    <metric>(item 2 rule-votes) / total-votes</metric>
  </experiment>
  <experiment name="SAH_11_equal" repetitions="1" runMetricsEveryStep="true">
    <setup>set votes1 500000 
set x-mean1 11
set y-mean1 0
set sd-1 4.5
set votes2 500000
set x-mean2 -11
set y-mean2 0
set sd-2 4.5
set votes3 0
set x-mean3 0
set y-mean3 0
set sd-3 1.0
set comfort-threshold 0
set fitness-alpha .5
set survival-threshold .3
set campaign-ticks 20
set new-party-rule "random-pick"
setup</setup>
    <go>go</go>
    <final>file-close</final>
    <timeLimit steps="20000"/>
    <metric>votes1</metric>
    <metric>x-mean1</metric>
    <metric>y-mean1</metric>
    <metric>sd-1</metric>
    <metric>votes2</metric>
    <metric>x-mean2</metric>
    <metric>y-mean2</metric>
    <metric>sd-2</metric>
    <metric>votes3</metric>
    <metric>x-mean3</metric>
    <metric>y-mean3</metric>
    <metric>sd-3</metric>
    <metric>comfort-threshold</metric>
    <metric>fitness-alpha</metric>
    <metric>survival-threshold</metric>
    <metric>campaign-ticks</metric>
    <metric>election</metric>
    <metric>voter-misery</metric>
    <metric>mean-eccentricity</metric>
    <metric>enp</metric>
    <metric>(item 0 rule-pcount)</metric>
    <metric>(item 0 rule-eccent)</metric>
    <metric>(item 0 rule-votes) / total-votes</metric>
    <metric>(item 1 rule-pcount)</metric>
    <metric>(item 1 rule-eccent)</metric>
    <metric>(item 1 rule-votes) / total-votes</metric>
    <metric>(item 2 rule-pcount)</metric>
    <metric>(item 2 rule-eccent)</metric>
    <metric>(item 2 rule-votes) / total-votes</metric>
  </experiment>
  <experiment name="SAH_12_equal" repetitions="1" runMetricsEveryStep="true">
    <setup>set votes1 500000 
set x-mean1 12
set y-mean1 0
set sd-1 4
set votes2 500000
set x-mean2 -12
set y-mean2 0
set sd-2 4
set votes3 0
set x-mean3 0
set y-mean3 0
set sd-3 1.0
set comfort-threshold 0
set fitness-alpha .5
set survival-threshold .3
set campaign-ticks 20
set new-party-rule "random-pick"
setup</setup>
    <go>go</go>
    <final>file-close</final>
    <timeLimit steps="20000"/>
    <metric>votes1</metric>
    <metric>x-mean1</metric>
    <metric>y-mean1</metric>
    <metric>sd-1</metric>
    <metric>votes2</metric>
    <metric>x-mean2</metric>
    <metric>y-mean2</metric>
    <metric>sd-2</metric>
    <metric>votes3</metric>
    <metric>x-mean3</metric>
    <metric>y-mean3</metric>
    <metric>sd-3</metric>
    <metric>comfort-threshold</metric>
    <metric>fitness-alpha</metric>
    <metric>survival-threshold</metric>
    <metric>campaign-ticks</metric>
    <metric>election</metric>
    <metric>voter-misery</metric>
    <metric>mean-eccentricity</metric>
    <metric>enp</metric>
    <metric>(item 0 rule-pcount)</metric>
    <metric>(item 0 rule-eccent)</metric>
    <metric>(item 0 rule-votes) / total-votes</metric>
    <metric>(item 1 rule-pcount)</metric>
    <metric>(item 1 rule-eccent)</metric>
    <metric>(item 1 rule-votes) / total-votes</metric>
    <metric>(item 2 rule-pcount)</metric>
    <metric>(item 2 rule-eccent)</metric>
    <metric>(item 2 rule-votes) / total-votes</metric>
  </experiment>
  <experiment name="SAH_7_unequal" repetitions="1" runMetricsEveryStep="true">
    <setup>set votes1 333333 
set x-mean1 7
set y-mean1 0
set sd-1 6.5
set votes2 666667
set x-mean2 -7
set y-mean2 0
set sd-2 6.5
set votes3 0
set x-mean3 0
set y-mean3 0
set sd-3 1.0
set comfort-threshold 0
set fitness-alpha .5
set survival-threshold .3
set campaign-ticks 20
set new-party-rule "random-pick"
setup</setup>
    <go>go</go>
    <final>file-close</final>
    <timeLimit steps="20000"/>
    <metric>votes1</metric>
    <metric>x-mean1</metric>
    <metric>y-mean1</metric>
    <metric>sd-1</metric>
    <metric>votes2</metric>
    <metric>x-mean2</metric>
    <metric>y-mean2</metric>
    <metric>sd-2</metric>
    <metric>votes3</metric>
    <metric>x-mean3</metric>
    <metric>y-mean3</metric>
    <metric>sd-3</metric>
    <metric>comfort-threshold</metric>
    <metric>fitness-alpha</metric>
    <metric>survival-threshold</metric>
    <metric>campaign-ticks</metric>
    <metric>election</metric>
    <metric>voter-misery</metric>
    <metric>mean-eccentricity</metric>
    <metric>enp</metric>
    <metric>(item 0 rule-pcount)</metric>
    <metric>(item 0 rule-eccent)</metric>
    <metric>(item 0 rule-votes) / total-votes</metric>
    <metric>(item 1 rule-pcount)</metric>
    <metric>(item 1 rule-eccent)</metric>
    <metric>(item 1 rule-votes) / total-votes</metric>
    <metric>(item 2 rule-pcount)</metric>
    <metric>(item 2 rule-eccent)</metric>
    <metric>(item 2 rule-votes) / total-votes</metric>
  </experiment>
  <experiment name="SAH_8_unequal" repetitions="1" runMetricsEveryStep="true">
    <setup>set votes1 333333 
set x-mean1 8
set y-mean1 0
set sd-1 6
set votes2 666667
set x-mean2 -8
set y-mean2 0
set sd-2 6
set votes3 0
set x-mean3 0
set y-mean3 0
set sd-3 1.0
set comfort-threshold 0
set fitness-alpha .5
set survival-threshold .3
set campaign-ticks 20
set new-party-rule "random-pick"
setup</setup>
    <go>go</go>
    <final>file-close</final>
    <timeLimit steps="20000"/>
    <metric>votes1</metric>
    <metric>x-mean1</metric>
    <metric>y-mean1</metric>
    <metric>sd-1</metric>
    <metric>votes2</metric>
    <metric>x-mean2</metric>
    <metric>y-mean2</metric>
    <metric>sd-2</metric>
    <metric>votes3</metric>
    <metric>x-mean3</metric>
    <metric>y-mean3</metric>
    <metric>sd-3</metric>
    <metric>comfort-threshold</metric>
    <metric>fitness-alpha</metric>
    <metric>survival-threshold</metric>
    <metric>campaign-ticks</metric>
    <metric>election</metric>
    <metric>voter-misery</metric>
    <metric>mean-eccentricity</metric>
    <metric>enp</metric>
    <metric>(item 0 rule-pcount)</metric>
    <metric>(item 0 rule-eccent)</metric>
    <metric>(item 0 rule-votes) / total-votes</metric>
    <metric>(item 1 rule-pcount)</metric>
    <metric>(item 1 rule-eccent)</metric>
    <metric>(item 1 rule-votes) / total-votes</metric>
    <metric>(item 2 rule-pcount)</metric>
    <metric>(item 2 rule-eccent)</metric>
    <metric>(item 2 rule-votes) / total-votes</metric>
  </experiment>
  <experiment name="SAH_9_unequal" repetitions="1" runMetricsEveryStep="true">
    <setup>set votes1 333333 
set x-mean1 9
set y-mean1 0
set sd-1 5.5
set votes2 666667
set x-mean2 -9
set y-mean2 0
set sd-2 5.5
set votes3 0
set x-mean3 0
set y-mean3 0
set sd-3 1.0
set comfort-threshold 0
set fitness-alpha .5
set survival-threshold .3
set campaign-ticks 20
set new-party-rule "random-pick"
setup</setup>
    <go>go</go>
    <final>file-close</final>
    <timeLimit steps="20000"/>
    <metric>votes1</metric>
    <metric>x-mean1</metric>
    <metric>y-mean1</metric>
    <metric>sd-1</metric>
    <metric>votes2</metric>
    <metric>x-mean2</metric>
    <metric>y-mean2</metric>
    <metric>sd-2</metric>
    <metric>votes3</metric>
    <metric>x-mean3</metric>
    <metric>y-mean3</metric>
    <metric>sd-3</metric>
    <metric>comfort-threshold</metric>
    <metric>fitness-alpha</metric>
    <metric>survival-threshold</metric>
    <metric>campaign-ticks</metric>
    <metric>election</metric>
    <metric>voter-misery</metric>
    <metric>mean-eccentricity</metric>
    <metric>enp</metric>
    <metric>(item 0 rule-pcount)</metric>
    <metric>(item 0 rule-eccent)</metric>
    <metric>(item 0 rule-votes) / total-votes</metric>
    <metric>(item 1 rule-pcount)</metric>
    <metric>(item 1 rule-eccent)</metric>
    <metric>(item 1 rule-votes) / total-votes</metric>
    <metric>(item 2 rule-pcount)</metric>
    <metric>(item 2 rule-eccent)</metric>
    <metric>(item 2 rule-votes) / total-votes</metric>
  </experiment>
  <experiment name="SAH_10_unequal" repetitions="1" runMetricsEveryStep="true">
    <setup>set votes1 333333 
set x-mean1 10
set y-mean1 0
set sd-1 5
set votes2 666667
set x-mean2 -10
set y-mean2 0
set sd-2 5
set votes3 0
set x-mean3 0
set y-mean3 0
set sd-3 1.0
set comfort-threshold 0
set fitness-alpha .5
set survival-threshold .3
set campaign-ticks 20
set new-party-rule "random-pick"
setup</setup>
    <go>go</go>
    <final>file-close</final>
    <timeLimit steps="20000"/>
    <metric>votes1</metric>
    <metric>x-mean1</metric>
    <metric>y-mean1</metric>
    <metric>sd-1</metric>
    <metric>votes2</metric>
    <metric>x-mean2</metric>
    <metric>y-mean2</metric>
    <metric>sd-2</metric>
    <metric>votes3</metric>
    <metric>x-mean3</metric>
    <metric>y-mean3</metric>
    <metric>sd-3</metric>
    <metric>comfort-threshold</metric>
    <metric>fitness-alpha</metric>
    <metric>survival-threshold</metric>
    <metric>campaign-ticks</metric>
    <metric>election</metric>
    <metric>voter-misery</metric>
    <metric>mean-eccentricity</metric>
    <metric>enp</metric>
    <metric>(item 0 rule-pcount)</metric>
    <metric>(item 0 rule-eccent)</metric>
    <metric>(item 0 rule-votes) / total-votes</metric>
    <metric>(item 1 rule-pcount)</metric>
    <metric>(item 1 rule-eccent)</metric>
    <metric>(item 1 rule-votes) / total-votes</metric>
    <metric>(item 2 rule-pcount)</metric>
    <metric>(item 2 rule-eccent)</metric>
    <metric>(item 2 rule-votes) / total-votes</metric>
  </experiment>
  <experiment name="SAH_11_unequal" repetitions="1" runMetricsEveryStep="true">
    <setup>set votes1 333333 
set x-mean1 11
set y-mean1 0
set sd-1 4.5
set votes2 666667
set x-mean2 -11
set y-mean2 0
set sd-2 4.5
set votes3 0
set x-mean3 0
set y-mean3 0
set sd-3 1.0
set comfort-threshold 0
set fitness-alpha .5
set survival-threshold .3
set campaign-ticks 20
set new-party-rule "random-pick"
setup</setup>
    <go>go</go>
    <final>file-close</final>
    <timeLimit steps="20000"/>
    <metric>votes1</metric>
    <metric>x-mean1</metric>
    <metric>y-mean1</metric>
    <metric>sd-1</metric>
    <metric>votes2</metric>
    <metric>x-mean2</metric>
    <metric>y-mean2</metric>
    <metric>sd-2</metric>
    <metric>votes3</metric>
    <metric>x-mean3</metric>
    <metric>y-mean3</metric>
    <metric>sd-3</metric>
    <metric>comfort-threshold</metric>
    <metric>fitness-alpha</metric>
    <metric>survival-threshold</metric>
    <metric>campaign-ticks</metric>
    <metric>election</metric>
    <metric>voter-misery</metric>
    <metric>mean-eccentricity</metric>
    <metric>enp</metric>
    <metric>(item 0 rule-pcount)</metric>
    <metric>(item 0 rule-eccent)</metric>
    <metric>(item 0 rule-votes) / total-votes</metric>
    <metric>(item 1 rule-pcount)</metric>
    <metric>(item 1 rule-eccent)</metric>
    <metric>(item 1 rule-votes) / total-votes</metric>
    <metric>(item 2 rule-pcount)</metric>
    <metric>(item 2 rule-eccent)</metric>
    <metric>(item 2 rule-votes) / total-votes</metric>
  </experiment>
  <experiment name="SAH_12_unequal" repetitions="1" runMetricsEveryStep="true">
    <setup>set votes1 333333 
set x-mean1 12
set y-mean1 0
set sd-1 4
set votes2 666667
set x-mean2 -12
set y-mean2 0
set sd-2 4
set votes3 0
set x-mean3 0
set y-mean3 0
set sd-3 1.0
set comfort-threshold 0
set fitness-alpha .5
set survival-threshold .3
set campaign-ticks 20
set new-party-rule "random-pick"
setup</setup>
    <go>go</go>
    <final>file-close</final>
    <timeLimit steps="20000"/>
    <metric>votes1</metric>
    <metric>x-mean1</metric>
    <metric>y-mean1</metric>
    <metric>sd-1</metric>
    <metric>votes2</metric>
    <metric>x-mean2</metric>
    <metric>y-mean2</metric>
    <metric>sd-2</metric>
    <metric>votes3</metric>
    <metric>x-mean3</metric>
    <metric>y-mean3</metric>
    <metric>sd-3</metric>
    <metric>comfort-threshold</metric>
    <metric>fitness-alpha</metric>
    <metric>survival-threshold</metric>
    <metric>campaign-ticks</metric>
    <metric>election</metric>
    <metric>voter-misery</metric>
    <metric>mean-eccentricity</metric>
    <metric>enp</metric>
    <metric>(item 0 rule-pcount)</metric>
    <metric>(item 0 rule-eccent)</metric>
    <metric>(item 0 rule-votes) / total-votes</metric>
    <metric>(item 1 rule-pcount)</metric>
    <metric>(item 1 rule-eccent)</metric>
    <metric>(item 1 rule-votes) / total-votes</metric>
    <metric>(item 2 rule-pcount)</metric>
    <metric>(item 2 rule-eccent)</metric>
    <metric>(item 2 rule-votes) / total-votes</metric>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
