globals [
  clock                     ;; clock denotes the days of simulation
  demand-today              ;; today's demand at both retailers
  colors                    ;; list of colors for plot pens
  stop-signal?              ;; if stop-signal? is true, the program will be stopped
]

breed [players player]     


;; there are two types of directed-links in this model
;; one for supply pipeline, one for demand placement
directed-link-breed [supply-links supply-link]  
                                                
supply-links-own [
  orders-filled          ;; a list standing for the shipping pipeline
  pair-demand-link       ;; the corresponding demand link with the same end nodes
]

directed-link-breed [demand-links demand-link]

demand-links-own [
  orders-placed          
  back-orders            ;; back-order for the end1 of the demand link
]



players-own [
  user-id
  role               ;; in this model, role can be "supplier", "distributor" or "retailer"
  number             ;; to distinguish players in the same echelon
  pen-color          ;; their pen-color on the plot
  
  on-hand            ;; on-hand inventory
  backlog            ;; total back-orders, equal to the sum or back-orders to each of its downstream nodes
  
  last-received
  finished?          ;; whether order placement is finished
  cost
  
  demand1            ;; demand from downstream node with number = 1 or 2
  demand2             
  
  last-shipped1      ;; amount shipped to downstream node with number = 1 or 2
  last-shipped2
  
  order1             ;; order to place to upstream node with number = 1 or 2
  order2
  
  ordered1           ;; order placed to upstream node with number = 1 or 2
  ordered2
]

;;
;; Startup procedure
;;
to startup
  hubnet-reset
  set-default-shape players "circle"
  set-default-shape links "arc"

end

to recruit  
  
  ;; we will recruit 4 players during this procedure
  
  while [hubnet-message-waiting?]                 
  [
    hubnet-fetch-message
    if hubnet-enter-message? [create-player]
    if hubnet-exit-message? [remove-player] 
  ]
  every 0.1 [display]
end


to go  
  listen-to-clients
  if stop-signal? [stop]         ;; if a stop signal is seen, we stop the program. 
                                 ;; this is usually because some players drop offline
  start-shipping
  every 0.1 [display]
end

to listen-to-clients
  while [ hubnet-message-waiting? ]
  [
    hubnet-fetch-message
    ifelse hubnet-enter-message?[                ;; during the game, new commers will be rejected
      hubnet-kick-client hubnet-message-source
    ][  
    ifelse hubnet-exit-message?
    [
      remove-player                           ;; if a player drops offline, then there are not enough players
      set stop-signal? true                   ;; to continue the game. we remove this player and send stop signal  
      ]
    [execute-command hubnet-message-tag]      
    ]
  
  ]
end

to execute-command [cmd]
  
  ;; players can control 3 things: 2 order sliders and 1 finish button
  
  ask players with [user-id = hubnet-message-source][
    if clock <= days-of-simulation [
      if hubnet-message-tag = "orders-to1" [set order1 hubnet-message]   
      if hubnet-message-tag = "orders-to2" [set order2 hubnet-message]
      if hubnet-message-tag = "finish" [
        place-order
      ]
    ]
  ]
end


to place-order
  if not finished? [    ;; order can be place for only once in a day
    set finished? true
    set ordered1 order1 ;; in the network supply chain, we can control 
    set ordered2 order2 ;; the order quantity to each upstream nodes
    
    hubnet-send user-id "finished?" finished?
    
    hubnet-send user-id "ordered1" order1
    
    ;; distributors only place order to 1 supplier
    ;; so nothing is displayed in "ordered2" for distributors
    
    if role = "retailer"[
      hubnet-send user-id "ordered2" order2  
    ]
    
  ]
end

to start-shipping
  ;; if all players have finished order placement
  ;; we conclude the day, and start shipping
  
  ifelse not any? players with [not finished?][
    place-order-to-up
    receive-order-from-up
    process-order-from-down
    summarize
    resize-shape
    set clock clock + 1               ;; proceed the day clock
    set demand-today daily-demand     ;; generate today's demand
    ask players with [role = "retailer" or role = "distributor"][
      set finished? false
      plot-cost
      plot-inventory              ;; plot cost and inventory
    ]
    update-to-clients
    update-order-info             ;; update parameters to the client windows
  ][
  update-order-info
  ]
end



to update-to-clients
  ask players with [role = "distributor" or role = "retailer"][
    hubnet-send user-id "ordered1" ordered1
    
    if role = "retailer"[    ;; again, distributors don't have "ordered2"
      hubnet-send user-id "ordered2" ordered2
    ]
    
    hubnet-send user-id "last-received" last-received
    hubnet-send user-id "finished?" finished?
    hubnet-send user-id "day" clock
    hubnet-send user-id "on-hand-inventory" on-hand
    hubnet-send user-id "back-orders" backlog
    hubnet-send user-id "cost" cost
  ]
  
  ask players with [role  = "distributor"][    
    set last-shipped1 [last orders-filled] of one-of my-out-supply-links with [[number] of end2 = 1] 
    set last-shipped2 [last orders-filled] of one-of my-out-supply-links with [[number] of end2 = 2] 
    hubnet-send user-id "last-shipped1" last-shipped1
    hubnet-send user-id "last-shipped2" last-shipped2
  ]
end

to update-order-info  
  ask players with [role = "distributor"][
    set demand1 [orders-placed] of one-of my-in-demand-links with [[number] of end1 = 1]  
    set demand2 [orders-placed] of one-of my-in-demand-links with [[number] of end1 = 2]
    hubnet-send user-id "demand1" demand1   
    hubnet-send user-id "demand2" demand2
  ]
  ask players with [role = "retailer"][
    set demand1 demand-today
    hubnet-send user-id "demand1" demand1
    hubnet-send user-id "demand2" ""       ;;retailers don't have "demand2"
  ]
end


;;
;; Inventory operations
;;

to place-order-to-up
  ask players with [role = "retailer"][
    let amount-order-to1 ordered1
    let amount-order-to2 ordered2
    foreach sort my-out-demand-links[   ;; set the order quantity to the correpsonding demand link
      ask ?[
        if [number] of end2 = 1 [set orders-placed amount-order-to1]
        if [number] of end2 = 2 [set orders-placed amount-order-to2]
      ]
    ]
  ]
  
  ask players with [role = "distributor"][
    let amount-order-to-supplier ordered1        ;; distributors are only order from the supplier
    ask one-of my-out-demand-links [set orders-placed amount-order-to-supplier]
  ]
end

to receive-order-from-up
  ask players [
    if role = "distributor" or role = "retailer"[    
      set last-received sum [first orders-filled] of my-in-supply-links  ;; pick out the first item in the pipeline
      ask my-in-supply-links [set orders-filled but-first orders-filled] ;; and remove it from the pipeline
      set on-hand on-hand + last-received                                ;; and add it to the inventory 
    ]
    if role = "supplier" [set on-hand 10000]                             ;; we assume that the supplier always has sufficient supply
  ]
end


to process-order-from-down
  ask players [
    
    let new-orders 0
    if role = "distributor" or role = "supplier"[                 ;; demand for suppliers and distributors are equal to
      set new-orders sum [orders-placed] of my-in-demand-links    ;; the sum of orders from downstream nodes
    ]
    if role = "retailer" [
      set new-orders demand-today             ;; demand for retailers is simply today's demand
      
    ]
    let orders-requested new-orders + backlog    ;; we need to satisfy both new orders and the back-orders
    let orders-to-ship min list orders-requested on-hand  ;; if not enough inventory, ship as many as we have
    
    set backlog max list 0 (backlog - on-hand + new-orders)  ;; unsatisfied quantity is considered back-order
    
    let rest-amount orders-to-ship
    
    foreach sort my-out-supply-links [      ;; since we are shipping to multiple downstream nodes
      ask ? [                               ;; if not enough inventory, we need to allocate the available stock
                                            ;; the quota of each downstream node is proportional to their requested quantity
                                            ;; considering both back-orders and new orders
        let quota sum [back-orders] of pair-demand-link + sum [orders-placed] of pair-demand-link 
        let ship-to-this-link 0
        if orders-requested > 0 [           
          set ship-to-this-link min list ceiling (quota * orders-to-ship / orders-requested) rest-amount
        ]
        set rest-amount rest-amount - ship-to-this-link
        set orders-filled lput ship-to-this-link orders-filled
        ask pair-demand-link [set back-orders max list 0 (quota - ship-to-this-link)]
      ]
    ]
    
    set on-hand on-hand - orders-to-ship    ;; reduce the on-hand inventory by the amount shipped
  ]
end

to summarize
  ask players with [role = "distributor" or role = "retailer"][
    let holding-cost on-hand * 0.5
    let back-order-cost backlog * 2
    set cost cost + holding-cost + back-order-cost    ;; summarize the day and calculate the cost
  ] 
end

to resize-shape
  ask players with [role = "distributor" or role = "retailer"][
    set size 0.2 * (sqrt on-hand)          ;; visualize the on-hand stock via size of the turtle
  ]
end



to create-player
  ifelse count players with [role != "supplier"] < 4 [   ;; if the number of players is less than 4
    create-players 1 [                                   ;; we can keep recruiting players
      set user-id hubnet-message-source         
      set label user-id                                  ;; label the players with their user-id first
      setxy (- max-pxcor + random max-pxcor) (- max-pycor + random max-pycor) ;; randomly place them in the world
    ]
  ][
  hubnet-kick-client hubnet-message-source    ;; if there are enough players, new commers are rejected
  ]
end

to remove-player
  ask players with [user-id = hubnet-message-source][die]
end

to setup
  set stop-signal? false      ;; reset the stop signal so that the program can go on
  layout
  label-players
  initialize
  resize-shape
  reset-plots
end

to layout        ;; layout the network supply chain
  ask players with [role != "supplier"] [set role "unassigned"]  ;; reset the roles but not removing the players
  ask links [die]        ;; eliminate all the links
  
  ask n-of 2 players with [role = "unassigned"][
    set role "distributor"        ;; assign 2 distributors
    set color blue 
  ]
  
  ask n-of 2 players with [role = "unassigned"][
    set role "retailer"           ;; assign 2 retailers
    set color green
  ]
  
  ask patch -10 0 [
    if not any? players-here[
      sprout-players 1 [          ;; only 1 supply with sufficient supply in the world
        set role "supplier"
        set color red
        set size 5
      ]
    ]
  ]
  
  ;; create links according to their roles
  ask players with [role = "retailer"] [
    create-demand-links-to players with [role = "distributor"]
    create-supply-links-from players with [role = "distributor"]
  ]
  ask players with [role = "distributor"] [
    create-demand-links-to players with [role = "supplier"]
    create-supply-links-from players with [role = "supplier"] 
  ]
  
  
  
  ask supply-links [   ;; define pair demand link, which has the same two ends
    set pair-demand-link demand-links with [end1 = [end2] of myself and end2 = [end1] of myself]
  ]
  
  ask players with [role = "distributor" or role = "retailer"][
    hubnet-send user-id "role" role   ;; inform players of their roles in the game
  ]
  
  
  
end

to label-players
  
  let distributors players with [role = "distributor"]
  let num-dist count distributors
  foreach sort distributors [
    ask ? [
      set number num-dist
      set num-dist num-dist - 1       ;; distinguish distributors by giving them different "number"
    ] 
  ]
  
  let retailers players with [role = "retailer"]
  let num-ret count retailers
  foreach sort retailers [
    ask ? [ 
      set number num-ret
      set num-ret num-ret - 1        ;; distinguish retailers by giving them different "number"
    ]
  ]
  
  ask players [
    set label word user-id "-SKU"    ;; relabel the players with their "user-id" and "number"
    set label word label number
  ] 
  
  ask players with [role = "distributor"][
    setxy 0 (-7 + (number - 1) * 14) 
  ] 
                                             ;; set distributors in the middle, retailers on the right
  ask players with [role = "retailer"][
    setxy 10 (-7 + (number - 1) * 14) 
  ]
  
  
  set colors [red blue green yellow]
  let index 0
  foreach sort players with [role = "distributor" or role = "retailer"][
    ask ? [
      set pen-color item index colors      ;; assign differnt pen-colors to players
      set index index + 1
    ]
  ]
end


to initialize
  set clock 0    ;; reset day clock
  set demand-today daily-demand      ;; generate today's demand
  ask players [
    
    set on-hand 100
    set backlog 0
    
    set last-received 0
    
    set finished? false
    if role = "supplier" [set finished? true] 
    
    set cost 0
    
    
    set last-shipped1 0
    set last-shipped2 0
    
    set order1 10
    set order2 10
    
    set ordered1 0
    set ordered2 0
    
    if role = "distributor"[   ;; distributors order from only 1 supplier
      set order2 ""            ;; thus no "ordered2"
      set ordered2 ""
      hubnet-send user-id "ordered2" ordered2
    ]
    
    if role = "retailer" [                   ;; retailers face only 1 external demand
      hubnet-send user-id "demand2" ""       ;; thus no "demand2"
      hubnet-send user-id "last-shipped2" ""
    ]
    
    
  ]
  ask supply-links [
    set orders-filled n-values lead-time [0]   ;; supply pipeline as list with "lead-time" elements
  ] 
  ask demand-links [
    set orders-placed 0
    set back-orders 0
  ]
  

  update-to-clients
  update-order-info
end

to-report daily-demand
  report random-poisson mean-for-poisson    ;; we use poisson distribution for the demand
end


;;
;; plot procedure
;;

to create-plot-pen [my-plot]
  set-current-plot my-plot
  create-temporary-plot-pen user-id   ;; name plot pens according to their user-id
  set-plot-pen-color pen-color        ;; set pen-color
end


to plot-cost
  set-current-plot "Total-cost"
  set-current-plot-pen user-id
  plot cost
end

to plot-inventory
  set-current-plot "On-hand-stock"
  set-current-plot-pen user-id
  plot on-hand
end

to reset-plots
  clear-all-plots
  ask players with [role = "distributor" or role = "retailer"][
    create-plot-pen "Total-cost"
    create-plot-pen "On-hand-stock"
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
204
7
544
368
16
16
10.0
1
10
1
1
1
0
0
0
1
-16
16
-16
16
1
1
1
ticks
30.0

BUTTON
26
18
94
52
NIL
recruit
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
113
18
177
52
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
24
252
196
285
lead-time
lead-time
1
10
2
1
1
NIL
HORIZONTAL

BUTTON
114
65
178
99
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
25
206
197
239
days-of-simulation
days-of-simulation
30
100
50
1
1
NIL
HORIZONTAL

PLOT
554
8
813
184
Total-cost
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS

PLOT
555
200
815
369
On-hand-stock
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS

SLIDER
22
307
194
341
mean-for-poisson
mean-for-poisson
0
50
20
1
1
NIL
HORIZONTAL

TEXTBOX
26
117
189
191
* Keep recruiting until you have 4 players\n* Press setup to establish the supply chain\n* Keep going to play the game
10
0.0
1

@#$#@#$#@
## WHAT IS IT?

In this model, we develp a simplified network supply chain. There are 1 supplier, 2 distributors, and 2 retailers. The supplier is assumed to always have sufficient supply. The distributors and retailers are played by clients. The players are to place order from upstream nodes and to ship to satisfy demands from downstream nodes. Their objective is to minimize the cost, which consists of the inventory holding cost and the penalty cost for back-orders. Questions to ask involve: How delays in the shipment can affect the entire supply chain? How the players at the same echelon will compete with the other? What is the best strategy to place orders?  

## HOW IT WORKS

There is a day clock in the model. Everyday a demand is imposed to the retailers. The job of the players is to place order to upstream nodes. In the network supply chain, a player can place orders to multiple upstream nodes. Thus in this model, we allow the players to allocate their orders. Once all players finished ordering, the program will start shipping and summarize the day. 

The players will put the requested order quantity into the supply pipelines. If there is not sufficient inventory, the unsatisfied quantity is counted as back-orders, which is to be fulfilled once inventory is avaialbe. In the network supply chain, a player may face multiple downstrean nodes. Thus in this model, we assume the shipping quantity is proportional to the requested order from each downstrean nodes.

The shipment takes certain lead time to arrive. Once the shipment arrives, the player will receive it into the inventory. 
  

## HOW TO USE IT

For the main interface:
First, press the "recruit" botton to recruit players. Once you have 4 players, you can unpress the "recruit" button. Additional players will be rejected.

Second, press "setup" button to establish the network supply chain and assign roles to players.

Last, press "go" button and start the game. You are allowed to press "setup" and reset the game without losing the current players.

The "days-of-simulation" is used to controal the rounds of simulation. The "lead-time" stands for the shipping delay of the supply pipeline. The daily demand is generated by random-poisson. The "mean-for-poisson" represents the mean of the poisson distribution.

The plots show the total cost and on-hand inventory of each player. 

For the client's window:
You can use the sliders "orders-to1" and "orders-to2" to adjust the order quantity you want to place to upstream node1 and node2. Press "finish" button once you decide to order.

After all players in the game have placed their orders. The day is concluded and a new day will begin.

## THINGS TO NOTICE

Players may want to find out the best order strategy that would minimize their cost.

Since competitors exist, it may be interesting to see how they will adjust their order strategy to compete with each other.

Also, will the retailers have loyaltiy to certain distributor?

## THINGS TO TRY

The observer can try to adjust "days-of-simulation" slider to see if the length of simulation would affect players order stategy.

The observer can also adjust "lead-time" slider to see if longer delays in supply will add difficulties to the players.

The observer can adjust "mean-for-poisson" to change to daily demand.

## EXTENDING THE MODEL

An interesting extension may be add the pricing mechanism into the model. That is to let the distributors and the retailers set the selling price. Then the players' objective become maximizing their own profit.

The current model involves only 2 distributors and 2 retailers. A flexible supply chain that enables the observer to adjust the number of distributors and retailers may be interesting.

The current model involves only 3 echelons: supplier, distributor, and retailer. More echelons can be added.

The current model allows full connection, e.g., each retailer can connect to all distributors. It would be interesting to explore what will happen in the limited connection situation.

## NETLOGO FEATURES

This is a HubNet version of the previously developed ABM network supply chain model by the author.

## RELATED MODELS

Wilensky, U. and Stroup, W. (2003). NetLogo HubNet Root Beer Game model. http://ccl.northwestern.edu/netlogo/models/HubNetRootBeerGame. Center for Connected Learning and Computer-Based Modeling, Northwestern Institute on Complex Systems, Northwestern University, Evanston, IL.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0.4
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
MONITOR
18
17
134
66
role
NIL
3
1

MONITOR
144
17
201
66
day
NIL
3
1

MONITOR
216
236
287
285
finished?
NIL
3
1

MONITOR
137
78
211
127
demand1
NIL
3
1

MONITOR
138
138
212
187
demand2
NIL
3
1

MONITOR
332
18
424
67
back-orders
NIL
3
1

MONITOR
18
80
124
129
last-shipped1
NIL
3
1

MONITOR
18
138
119
187
last-shipped2
NIL
3
1

MONITOR
106
235
203
284
last-received
NIL
3
1

MONITOR
208
18
322
67
on-hand-inventory
NIL
3
1

MONITOR
222
78
295
127
ordered1
NIL
3
1

MONITOR
224
139
297
188
ordered2
NIL
3
1

SLIDER
316
84
488
117
orders-to1
orders-to1
0
100
10
1
1
NIL
HORIZONTAL

SLIDER
314
150
488
183
orders-to2
orders-to2
0
100
10
1
1
NIL
HORIZONTAL

MONITOR
431
18
488
67
cost
NIL
3
1

BUTTON
314
238
380
271
finish
NIL
NIL
1
T
OBSERVER
NIL
NIL

TEXTBOX
226
198
393
226
** distributors don't have order2
10
0.0
1

TEXTBOX
20
198
208
226
** retailers don't have demand 2
10
0.0
1

VIEW
508
19
794
305
0
0
0
1
1
1
1
1
0
1
1
1
-16
16
-16
16

@#$#@#$#@
default
0.0
-0.2 0 1.0 0.0
0.0 1 1.0 0.0
0.2 0 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

arc
2.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Polygon -7500403 true true 150 150 105 225 180 225 195 225 150 150

@#$#@#$#@
0
@#$#@#$#@
