<?xml version="1.0" encoding="utf-8"?>
<model version="NetLogo 7.0.0" snapToGrid="false">
  <code><![CDATA[globals [
  infection-chance  ;; The chance out of 100 that an infected person will pass on
                    ;;   infection during one week of couplehood.
  symptoms-show     ;; How long a person will be infected before symptoms occur
                    ;;   which may cause the person to get tested.
  slider-check-1    ;; Temporary variables for slider values, so that if sliders
  slider-check-2    ;;   are changed on the fly, the model will notice and
  slider-check-3    ;;   change people's tendencies appropriately.
  slider-check-4
]

turtles-own [
  infected?          ;; If true, the person is infected.  It may be known or unknown.
  known?             ;; If true, the infection is known (and infected? must also be true).
  infection-length   ;; How long the person has been infected.
  coupled?           ;; If true, the person is in a sexually active couple.
  couple-length      ;; How long the person has been in a couple.
  ;; the next four values are controlled by sliders
  commitment         ;; How long the person will stay in a couple-relationship.
  coupling-tendency  ;; How likely the person is to join a couple.
  condom-use         ;; The percent chance a person uses protection.
  test-frequency     ;; Number of times a person will get tested per year.
  partner            ;; The person that is our current partner in a couple.
  treated?           ;; whether those who are infected and know it are recieving treatment.
]

;;;
;;; SETUP PROCEDURES
;;;

to setup
  clear-all
  setup-globals
  setup-people
  reset-ticks
end

to setup-globals
  set infection-chance 50    ;; if you have unprotected sex with an infected partner,
                             ;; you have a 50% chance of being infected
  set symptoms-show 200.0    ;; symptoms show up 200 weeks after infection
  set slider-check-1 average-commitment
  set slider-check-2 average-coupling-tendency
  set slider-check-3 average-condom-use
  set slider-check-4 average-test-frequency
end

;; Create carrying-capacity number of people half are righty and half are lefty
;;   and some are sick.  Also assigns colors to people with the ASSIGN-COLORS routine.

to setup-people
  create-turtles initial-people
    [ setxy random-xcor random-ycor
      set known? false
      set coupled? false
      set treated? false
      set partner nobody
      ifelse random 2 = 0
        [ set shape "person righty" ]
        [ set shape "person lefty" ]
      ;; 2.5% of the people start out infected, but they don't know it
      set infected? (who < initial-people * 0.025)
      if infected?
        [ set infection-length random-float symptoms-show ]
      assign-commitment
      assign-coupling-tendency
      assign-condom-use
      assign-test-frequency
      assign-color
     
  ]
end

;; Different people are displayed in 3 different colors depending on health
;; green is not infected
;; blue is infected but doesn't know it
;; red is infected and knows it
;; violet is infected and knows it, but has treatment

to assign-color
  ifelse not infected? [
    set color green
  ] [
    ifelse treated? [
      set color violet
    ] [
      ifelse known? [ set color red ] [ set color blue ]
    ]
  ]
end


;; The following four procedures assign core turtle variables.  They use
;; the helper procedure RANDOM-NEAR so that the turtle variables have an
;; approximately "normal" distribution around the average values set by
;; the sliders.

to assign-commitment  ;; turtle procedure
  set commitment random-near average-commitment
end

to assign-coupling-tendency  ;; turtle procedure
  set coupling-tendency random-near average-coupling-tendency
end

to assign-condom-use  ;; turtle procedure
  set condom-use random-near average-condom-use
end

to assign-test-frequency  ;; turtle procedure
  set test-frequency random-near average-test-frequency
end

to-report random-near [center]  ;; turtle procedure
  let result 0
  repeat 40
    [ set result (result + random-float center) ]
  report result / 20
end

;;;
;;; GO PROCEDURES
;;;

to go
  if all? turtles [known?]
    [ stop ]
  check-sliders
  ask turtles
    [ if infected?
        [ set infection-length infection-length + 1 ]
      if coupled?
        [ set couple-length couple-length + 1 ] ]
  ask turtles
    [ if not coupled?
        [ move ] ]
  ;; Righties are always the ones to initiate mating.  This is purely
  ;; arbitrary choice which makes the coding easier.
  ask turtles
    [ if not coupled? and shape = "person righty" and (random-float 10.0 < coupling-tendency)
        [ couple ] ]
  ask turtles [ uncouple ]
  ask turtles [ infect ]
  ask turtles [ test ]
  ask turtles [ assign-color ]
  tick
end

;; Each tick a check is made to see if sliders have been changed.
;; If one has been, the corresponding turtle variable is adjusted

to check-sliders
  if (slider-check-1 != average-commitment)
    [ ask turtles [ assign-commitment ]
      set slider-check-1 average-commitment ]
  if (slider-check-2 != average-coupling-tendency)
    [ ask turtles [ assign-coupling-tendency ]
      set slider-check-2 average-coupling-tendency ]
  if (slider-check-3 != average-condom-use)
    [ ask turtles [ assign-condom-use ]
      set slider-check-3 average-condom-use ]
  if (slider-check-4 != average-test-frequency )
    [ ask turtles [ assign-test-frequency ]
      set slider-check-4 average-test-frequency ]
end

;; People move about at random.

to move  ;; turtle procedure
  rt random-float 360
  fd 1
end

;; People have a chance to couple depending on their tendency to have sex and
;; if they meet.  To better show that coupling has occurred, the patches below
;; the couple turn gray.

to couple  ;; turtle procedure -- righties only!
  let potential-partner one-of (turtles-at -1 0)
                          with [not coupled? and shape = "person lefty"]
  if potential-partner != nobody
    [ if random-float 10.0 < [coupling-tendency] of potential-partner
      [ set partner potential-partner
        set coupled? true
        ask partner [ set coupled? true ]
        ask partner [ set partner myself ]
        move-to patch-here ;; move to center of patch
        ask potential-partner [move-to patch-here] ;; partner moves to center of patch
        set pcolor gray - 3
        ask (patch-at -1 0) [ set pcolor gray - 3 ] ] ]
end

;; If two peoples are together for longer than either person's commitment variable
;; allows, the couple breaks up.

to uncouple  ;; turtle procedure
  if coupled? and (shape = "person righty")
    [ if (couple-length > commitment) or
         ([couple-length] of partner) > ([commitment] of partner)
        [ set coupled? false
          set couple-length 0
          ask partner [ set couple-length 0 ]
          set pcolor black
          ask (patch-at -1 0) [ set pcolor black ]
          ask partner [ set partner nobody ]
          ask partner [ set coupled? false ]
          set partner nobody ] ]
end

;; Infection can occur if either person is infected, but the infection is unknown.
;; This model assumes that people with known infections will continue to couple,
;; but will automatically practice safe sex, regardless of their condom-use tendency.
;; Note also that for condom use to occur, both people must want to use one.  If
;; either person chooses not to use a condom, infection is possible.  Changing the
;; primitive to AND in the third line will make it such that if either person
;; wants to use a condom, infection will not occur.
;; if treatment-enabled is true, infection-chance decreases by 90%.

to infect  ;; turtle procedure
  if coupled? and infected? [
    
    ;; Determine infection chance based on treatment
    let effective-chance (infection-chance)
    if treated? [
      set effective-chance (infection-chance * 0.1)  ;; 90% reduction if treated
    ]

    ;; Check condom-use
    if random-float 10 > condom-use or
       random-float 10 > ([condom-use] of partner) [
       
       ;; Attempt transmission
       if random-float 100 < effective-chance [
         ask partner [ set infected? true ]
       ]
    ]
  ]
end

;; People have a tendency to check out their health status based on a slider value.
;; This tendency is checked against a random number in this procedure. However, after being infected for
;; some amount of time called SYMPTOMS-SHOW, there is a 5% chance that the person will
;; become ill and go to a doctor and be tested even without the tendency to check.

to test  ;; turtle procedure
  if random-float 52 < test-frequency [
    if infected? [
      set known? true
      if treatment-enabled [
        set treated? true
      ]
    ]
  ]

  if infection-length > symptoms-show [
    if random-float 100 < 5 [
      set known? true
      if treatment-enabled [
        set treated? true
      ]
    ]
  ]
end

;;;
;;; MONITOR PROCEDURES
;;;

    
to-report %infected
  ifelse any? turtles
    [ report (count turtles with [infected?] / count turtles) * 100 ]
    [ report 0 ]
end


; Copyright 1997 Uri Wilensky.
; See Info tab for full copyright and license.]]></code>
  <widgets>
    <view x="305" wrappingAllowedX="true" y="10" frameRate="30.0" minPycor="-12" height="429" showTickCounter="true" patchSize="17.0" fontSize="10" wrappingAllowedY="true" width="429" tickCounterLabel="weeks" maxPycor="12" updateMode="1" maxPxcor="12" minPxcor="-12"></view>
    <button x="96" y="98" height="35" disableUntilTicks="true" forever="true" kind="Observer" width="83" display="go">go</button>
    <monitor x="184" precision="2" y="91" height="60" fontSize="11" width="100" display="% infected">%infected</monitor>
    <slider x="7" step="1" y="37" max="500" width="277" display="initial-people" height="50" min="50" direction="Horizontal" default="300.0" variable="initial-people"></slider>
    <slider x="7" step="1" y="211" max="200" width="277" display="average-commitment" height="50" min="1" direction="Horizontal" default="1.0" variable="average-commitment" units="weeks"></slider>
    <slider x="7" step="1" y="159" max="10" width="277" display="average-coupling-tendency" height="50" min="0" direction="Horizontal" default="5.0" variable="average-coupling-tendency"></slider>
    <slider x="7" step="1" y="263" max="10" width="277" display="average-condom-use" height="50" min="0" direction="Horizontal" default="0.0" variable="average-condom-use"></slider>
    <slider x="7" step="0.01" y="315" max="2" width="277" display="average-test-frequency" height="50" min="0" direction="Horizontal" default="0.0" variable="average-test-frequency" units="times/year"></slider>
    <switch x="8" y="374" height="40" on="true" variable="treatment-enabled" width="273" display="treatment-enabled"></switch>
    <plot x="6" autoPlotX="true" yMax="350.0" autoPlotY="true" yAxis="people" y="426" xMin="0.0" height="206" legend="true" xMax="52.0" yMin="0.0" width="277" xAxis="weeks" display="Populations">
      <setup>set-plot-y-range 0 (initial-people + 50)</setup>
      <update></update>
      <pen interval="1.0" mode="0" display="HIV-" color="-10899396" legend="true">
        <setup></setup>
        <update>plot count turtles with [not infected?]</update>
      </pen>
      <pen interval="1.0" mode="0" display="HIV+" color="-2674135" legend="true">
        <setup></setup>
        <update>plot count turtles with [known?]</update>
      </pen>
      <pen interval="1.0" mode="0" display="HIV?" color="-13345367" legend="true">
        <setup></setup>
        <update>plot count turtles with [infected?] - count turtles with [known?]</update>
      </pen>
      <pen interval="1.0" mode="0" display="HIV+ Treated" color="-8630108" legend="true">
        <setup></setup>
        <update>plot count turtles with [treated?]</update>
      </pen>
    </plot>
    <button x="12" y="98" height="35" disableUntilTicks="false" forever="false" kind="Observer" width="83" display="setup">setup</button>
    <monitor x="833" precision="17" y="22" height="60" fontSize="11" width="100" display="infection%">infection-chance</monitor>
  </widgets>
  <info><![CDATA[## WHAT IS IT?

This model simulates the spread of the human immunodeficiency virus (HIV), via sexual transmission, through a small isolated human population.  It therefore illustrates the effects of certain sexual practices across a population.

As is well known now, HIV is spread in a variety of ways of which sexual contact is only one.  HIV can also be spread by needle-sharing among injecting drug users, through blood transfusions (although this has become very uncommon in countries like the United States in which blood is screened for HIV antibodies), or from HIV-infected women to their babies either before or during birth, or afterwards through breast-feeding.  This model focuses only on the spread of HIV via sexual contact.

The model examines the emergent effects of four aspects of sexual behavior.  The user controls the population's tendency to practice abstinence, the amount of time an average "couple" in the population will stay together, the population's tendency to use condoms, and the population's tendency to get tested for HIV.  Exploration of the first and second variables may illustrate how changes in sexual mores in our society have contributed to increases in the prevalence of sexually transmitted diseases, while exploration of the third and fourth may provide contemporary solutions to the problem.

By allowing the user to control how often an average individual will choose to be tested, the user can explore an important dimension of HIV's threat to public health.  Because the virus does not make itself immediately known in its host, individuals are often infected for some time before a test or immune deficiency symptoms (which leads to a test) identifies them as such.  Regular identification of individuals infected by the virus could have significant public health impacts if knowledge of the infection positively affected sexual behaviors.  This model explores this possibility by making all individuals who know of their positive HIV status always practice safe sex.

## HOW IT WORKS

The model uses "couples" to represent two people engaged in sexual relations.  Individuals wander around the world when they are not in couples.  Upon coming into contact with a suitable partner, there is a chance the two individuals will "couple" together.  When this happens, the two individuals no longer move about, they stand next to each other holding hands as a representation of two people in a sexual relationship.

The presence of the virus in the population is represented by the colors of individuals. Three colors are used: green individuals are uninfected, blue individuals are infected but their infection is unknown, and red individuals are infected and their infection is known.

## HOW TO USE IT

The SETUP button creates individuals with particular behavioral tendencies according to the values of the interface's five sliders (described below).  Once the simulation has been setup, you are now ready to run it, by pushing the GO button.  GO starts the simulation and runs it continuously until GO is pushed again.  During a simulation initiated by GO, adjustments in sliders can affect the behavioral tendencies of the population.

A monitor shows the percent of the population that is infected by HIV.  In this model each time-step is considered one week; the number of weeks that have passed is shown in the toolbar.

Here is a summary of the sliders in the model.  They are explained in more detail below.

- INITIAL-PEOPLE: How many people simulation begins with.
- AVERAGE-COUPLING-TENDENCY: General likelihood member of population has sex (0--10).
- AVERAGE-COMMITMENT: How many weeks sexual relationships typically lasts (0--200).
- AVERAGE-CONDOM-USE: General chance member of population uses a condom. (0--10).
- AVERAGE-TEST-FREQUENCY: Average frequency member of population will check their HIV status in a 1-year time period (0--2).

The total number of individuals in the simulation is controlled by the slider INITIAL-PEOPLE (initialized to vary between 50--500), which must be set before SETUP is pushed.

During the model's setup procedures, all individuals are given "tendencies" according to four additional sliders.  These tendencies are generally assigned in a normal distribution, so that, for instance, if a tendency slider is set at 8, the most common value for that tendency in the population will be 8.  Less frequently, individuals will have values 7 or 9 for that tendency, and even less frequently will individuals have values 6 or 10 (and so on).

The slider AVERAGE-COUPLING-TENDENCY (0--10) determines the tendency of the individuals to become involved in couples (as stated earlier, all couples are presumed to be sexually active). When the AVERAGE-COUPLING-TENDENCY slider is set at zero, coupling is unlikely, although (because tendencies are assigned in a normal distribution) it is still possible.  Note that when deciding to couple, both individuals involved must be "willing" to do so, so high coupling tendencies in two individuals are actually compounded (i.e. two individuals with a 50% chance of coupling actually only have a 25% of coupling in a given tick).

The slider AVERAGE-COMMITMENT (1--200) determines how long individuals are likely to stay in a couple (in weeks).  Again, the tendencies of both individuals in a relationship are considered; the relationship only lasts as long as is allowed by the tendency of the partner with a shorter commitment tendency.

The slider AVERAGE-CONDOM-USE (0--10) determines the tendency of individuals in the population to practice safe sex.  If an individual uses a condom, it is assumed in this model that no infection by HIV is possible.  Note that this tendency (like the others) is probabilistic at several levels.  For instance, when AVERAGE-CONDOM-USE is set to 9, most of the individuals have a CONDOM-USE value of 9, although some have CONDOM-USE values of 8 or 10, and fewer yet have CONDOM-USE values of 7 or 11 (11 would be off-scale and the same as 10 for all practical purposes).  Also, an individual with a CONDOM-USE value of 9 will still sometimes not choose to use a condom (10% of the time, roughly). Simulation of condom-use is further complicated by the fact that if one partner "wants" to use a condom while the other partner does not, the couple does not use condoms.  This characteristic of the model is representative of the dynamics of some sexual relations, but not others.  The decision was somewhat arbitrary, and the user is invited to play with this characteristic and others in the "Extending the Model" section of this tab.

The slider AVERAGE-TEST-FREQUENCY (0--2) is the final slider of the interface.  It determines the average frequency of an individual to get tested for HIV in a one-year time span.  Set at 1.0, the average person will get tested for HIV about once a year.  Set at 0.2, the average individual will only get tested about every five years.  This tendency has significant impact because the model assumes that individuals who know that they are infected always practice safe sex, even if their tendency as healthy individuals was different.  Again, this characteristic of the model is only represented of the behaviors of some individuals.  The model was designed in this way to highlight the public health effects associated with frequent testing *and* appropriate responses to knowledge of one's HIV status.  To explore the impact of alternative behaviors on public health, the model code can be changed relatively painlessly.  These changes are described in the section, "Extending the Model."

The model's plot shows the total number of uninfected individuals (green), infected individuals whose positive status is not known (blue), and infected individuals whose positive status is known (red).

## THINGS TO NOTICE

Set the INITIAL-PEOPLE slider to 300, AVERAGE-COUPLING-TENDENCY to 10, AVERAGE-COMMITMENT to 100, and the other two sliders to 0. Push SETUP and then push GO. Notice that many individuals rapidly pair up into stationary "couples", with the patches behind them turned a dark gray.  These couples represent sexual activity between the two individuals.  Individuals that continue to move about (and do not have a gray patch behind them) are not engaging in sexual relations.  With RELATIONSHIP-DURATION set to 100, an average individual will have monogamous sexual relations with a partner for about 100 weeks (approximately 2 years) before ending the sexual relationship and searching out a new partner.

Stop the simulation (by pressing the GO button once again), move the AVERAGE-COUPLING-TENDENCY slider to 0, push SETUP, and start the simulation again (with the GO button).  Notice that this time, couples are not forming.  With a low COUPLING-TENDENCY, individuals do not choose to have sex, which in this model is depicted by the graphical representation of couplehood.  Notice that with these settings, HIV does not typically spread. However, spreading could happen since the population's tendency is set according to a normal distribution and a few people will probably have COUPLING-TENDENCY values above 0 at this setting.

Again reset the simulation, this time with AVERAGE-COUPLING-TENDENCY set back to 10 and AVERAGE-COMMITMENT set to 1.  Notice that while individuals do not stand still next to each other for any noticeable length of time, coupling is nevertheless occurring.  This is indicated by the occasional flash of dark gray behind individuals that are briefly next to one another.  Notice that the spread of HIV is actually faster when RELATIONSHIP-DURATION is very short compared to when it is very long.

Now run a simulation with AVERAGE-COMMITMENT equal to 1, AVERAGE-COUPLING-TENDENCY set to 1, AVERAGE-CONDOM-USE set to 10, and AVERAGE-TEST-FREQUENCY set to 1.0. With negligible couple formation and high condom use anyway, there should be no spread of HIV.  Notice how pale blue "infection unknown" individuals turn red much quicker in this simulation.  When the individuals turn red, they know their HIV status.  Some individuals turn red because they have been infected for long enough that they develop symptoms, which alerts them to the need for an HIV test.  Others become red because they choose to be tested.  With AVERAGE-TEST-FREQUENCY set to 1.0, healthy individuals are also being tested, but their color does not change since the tests come back negative.

When all the individuals in the simulation are either green or red, change the sliders without stopping the simulation.  Set AVERAGE-COUPLING-TENDENCY to 10, AVERAGE-COMMITMENT to 100, AVERAGE-CONDOM-USE to 0, and AVERAGE-TEST-FREQUENCY to 0.  Notice that despite the immediate formation of couples and the fact that condom-use tendency is presumably very low, some between healthy (green) individuals and infected (red) individuals, no spread of HIV occurs.  This is because while the model expects HIV positive individuals to continue to engage in sexual relations, it presumes that those individuals will always use condoms and that the condoms will always work.  The rationale for this design is discussed briefly in the "What is it?" section of this tab.

Finally, set INITIAL-PEOPLE to 500 to notice that couples can form on top of each other.  Watch the simulation until you see individuals by themselves, but standing still and with a gray patch behind them indicating coupling.  Underneath one of its neighbors, is the individuals partner.  This apparent bug in the program is necessary because individuals need to be able to couple fairly freely.  If space constraints prohibited coupling, unwanted emergent behavior would occur with high population densities.

Couples are formed by a partnership of "righty" and "lefty" shapes which is not immediately noticeable.  These shapes are not intended to represent genders in any fashion, but merely to provide a useful way to depict couple graphics. In order for the hands of a righty and lefty to match up, both must be off-centered in their patch.  Without this feature, two couples next to each other would appear to be a line of four individuals (instead of two groups of two).  It is intended that the differences between righty and lefty shapes not be especially apparent in order to prevent unintended associations with gender.  Any righty or lefty shape can be thought of as male or female or neither.

## THINGS TO TRY

Run a number of experiments with the GO button to find out the effects of different variables on the spread of HIV.  Try using good controls in your experiment.  Good controls are when only one variable is changed between trials.  For instance, to find out what effect the average duration of a relationship has, run four experiments with the AVERAGE-COMMITMENT slider set at 1 the first time, 2 the second time, 10 the third time, and 50 the last.  How much does the prevalence of HIV increase in each case?  Does this match your expectations?

Are the effects of some slider variables mediated by the effects of others?  For instance, if lower AVERAGE-COMMITMENT values seem to increase the spread of HIV when all other sliders are set at 0, does the same thing occur if all other sliders are set at 10?  You can run many experiments to test different hypotheses regarding the emergent public health effects associated with individual sexual behaviors.

## EXTENDING THE MODEL

Like all computer simulations of human behaviors, this model has necessarily simplified its subject area substantially.  The model therefore provides numerous opportunities for extension:

The model depicts sexual activity as two people standing next to each other.  This suggests that all couples have sex and that abstinence is only practiced in singlehood.  The model could be changed to reflect a more realistic view of what couples are.  Individuals could be in couples without having sex.  To show sex, then, a new graphical representation would have to be employed.  Perhaps sex could be symbolized by having the patches beneath the couple flash briefly to a different color.

The model does not distinguish between genders.  This is an obvious over-simplification chosen because making an exclusively heterosexual model was untenable while making one that included a variety of sexual preferences might have distracted from the public health issues which it was designed to explore.  However, extending the model by adding genders would make the model more realistic.

The model assumes that individuals who are aware that they are infected always practice safe sex.  This portrayal of human behavior is clearly not entirely realistic, but it does create interesting emergent behavior that has genuine relevance to certain public health debate.  However, an interesting extension of the model would be to change individuals' reactions to knowledge of HIV status.

The model assumes that condom use is always 100% effective.  In fact, responsible condom use is actually slightly less than ideal protection from the transmission of HIV.  Add a line of code to the INFECT procedure to check for a slight random chance that a particular episode of condom-use is not effective.  Another change that can be made in the INFECT procedure regards a couple's choice of condom-use.  In this model, when the two partners of a couple "disagree" about whether or not to use a condom, the partner that doesn't wish to use a condom makes the choice.  The opposite possibility is also valid.

Finally, certain significant changes can easily be made in the model by simply changing the value of certain global variables in the procedure SETUP-GLOBALS.  Two variables set in this procedure that are especially worthy of investigation are INFECTION-CHANCE and SYMPTOMS-SHOW.  The former controls what chance an infection has of spreading from an infected to an uninfected partner if no protection is used.  The variable is currently set to 50, meaning that in a week of sexual relations, the chance of infection occurring is 50%.  It is not clear at this time how realistic that figure is. SYMPTOMS-SHOW is the variable that controls how long, on average, a person will have the HIV virus before symptoms begin to appear, alerting that person to the presence of some health problem.  It is currently set to 200 (weeks) in this model.

## NETLOGO FEATURES

Notice that the four procedures that assign the different tendencies generate many small random numbers and add them together.  This produces a normal distribution of tendency values.  A random number between 0 and 100 is as likely to be 1 as it is to be 99. However, the sum of 20 numbers between 0 and 5 is much more likely to be 50 than it is to be 99.

Notice that the global variables SLIDER-CHECK-1, SLIDER-CHECK-2, and so on are assigned with the values of the various sliders so that the model can check the sliders against the variables while the model is running.  Every time-step, a slider's value is checked against a global variable that holds the value of what the slider's value was the time-step before.  If the slider's current value is different than the global variable, NetLogo knows to call procedures that adjust the population's tendencies.  Otherwise, those adjustment procedures are not called.

## CREDITS AND REFERENCES

Special thanks to Steve Longenecker for model development.

## HOW TO CITE

If you mention this model or the NetLogo software in a publication, we ask that you include the citations below.

For the model itself:

* Wilensky, U. (1997).  NetLogo HIV model.  http://ccl.northwestern.edu/netlogo/models/HIV.  Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

Please cite the NetLogo software as:

* Wilensky, U. (1999). NetLogo. http://ccl.northwestern.edu/netlogo/. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

## COPYRIGHT AND LICENSE

Copyright 1997 Uri Wilensky.

![CC BY-NC-SA 3.0](http://ccl.northwestern.edu/images/creativecommons/byncsa.png)

This model is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 3.0 License.  To view a copy of this license, visit https://creativecommons.org/licenses/by-nc-sa/3.0/ or send a letter to Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.

NetLogo itself is free software for non-commercial use under the terms of the GNU General Public License ([see full license information here](https://ccl.northwestern.edu/netlogo/docs/copyright.html)).

To inquire about commercial licenses for either NetLogo or specific models from the models library, please contact netlogo-commercial-admin@ccl.northwestern.edu.

This model was created as part of the project: CONNECTED MATHEMATICS: MAKING SENSE OF COMPLEX PHENOMENA THROUGH BUILDING OBJECT-BASED PARALLEL MODELS (OBPML).  The project gratefully acknowledges the support of the National Science Foundation (Applications of Advanced Technologies Program) -- grant numbers RED #9552950 and REC #9632612.

This model was converted to NetLogo as part of the projects: PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN CLASSROOMS and/or INTEGRATED SIMULATION AND MODELING ENVIRONMENT. The project gratefully acknowledges the support of the National Science Foundation (REPP & ROLE programs) -- grant numbers REC #9814682 and REC-0126227. Converted from StarLogoT to NetLogo, 2001.

<!-- 1997 2001 -->]]></info>
  <turtleShapes>
    <shape name="default" rotatable="true" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="5"></point>
        <point x="40" y="250"></point>
        <point x="150" y="205"></point>
        <point x="260" y="250"></point>
      </polygon>
    </shape>
    <shape name="airplane" rotatable="true" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="0"></point>
        <point x="135" y="15"></point>
        <point x="120" y="60"></point>
        <point x="120" y="105"></point>
        <point x="15" y="165"></point>
        <point x="15" y="195"></point>
        <point x="120" y="180"></point>
        <point x="135" y="240"></point>
        <point x="105" y="270"></point>
        <point x="120" y="285"></point>
        <point x="150" y="270"></point>
        <point x="180" y="285"></point>
        <point x="210" y="270"></point>
        <point x="165" y="240"></point>
        <point x="180" y="180"></point>
        <point x="285" y="195"></point>
        <point x="285" y="165"></point>
        <point x="180" y="105"></point>
        <point x="180" y="60"></point>
        <point x="165" y="15"></point>
      </polygon>
    </shape>
    <shape name="arrow" rotatable="true" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="0"></point>
        <point x="0" y="150"></point>
        <point x="105" y="150"></point>
        <point x="105" y="293"></point>
        <point x="195" y="293"></point>
        <point x="195" y="150"></point>
        <point x="300" y="150"></point>
      </polygon>
    </shape>
    <shape name="box" rotatable="false" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="285"></point>
        <point x="285" y="225"></point>
        <point x="285" y="75"></point>
        <point x="150" y="135"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="135"></point>
        <point x="15" y="75"></point>
        <point x="150" y="15"></point>
        <point x="285" y="75"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="15" y="75"></point>
        <point x="15" y="225"></point>
        <point x="150" y="285"></point>
        <point x="150" y="135"></point>
      </polygon>
      <line endX="150" startY="285" marked="false" color="255" endY="135" startX="150"></line>
      <line endX="15" startY="135" marked="false" color="255" endY="75" startX="150"></line>
      <line endX="285" startY="135" marked="false" color="255" endY="75" startX="150"></line>
    </shape>
    <shape name="bug" rotatable="true" editableColorIndex="0">
      <circle x="96" y="182" marked="true" color="-1920102913" diameter="108" filled="true"></circle>
      <circle x="110" y="127" marked="true" color="-1920102913" diameter="80" filled="true"></circle>
      <circle x="110" y="75" marked="true" color="-1920102913" diameter="80" filled="true"></circle>
      <line endX="80" startY="100" marked="true" color="-1920102913" endY="30" startX="150"></line>
      <line endX="220" startY="100" marked="true" color="-1920102913" endY="30" startX="150"></line>
    </shape>
    <shape name="butterfly" rotatable="true" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="165"></point>
        <point x="209" y="199"></point>
        <point x="225" y="225"></point>
        <point x="225" y="255"></point>
        <point x="195" y="270"></point>
        <point x="165" y="255"></point>
        <point x="150" y="240"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="165"></point>
        <point x="89" y="198"></point>
        <point x="75" y="225"></point>
        <point x="75" y="255"></point>
        <point x="105" y="270"></point>
        <point x="135" y="255"></point>
        <point x="150" y="240"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="139" y="148"></point>
        <point x="100" y="105"></point>
        <point x="55" y="90"></point>
        <point x="25" y="90"></point>
        <point x="10" y="105"></point>
        <point x="10" y="135"></point>
        <point x="25" y="180"></point>
        <point x="40" y="195"></point>
        <point x="85" y="194"></point>
        <point x="139" y="163"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="162" y="150"></point>
        <point x="200" y="105"></point>
        <point x="245" y="90"></point>
        <point x="275" y="90"></point>
        <point x="290" y="105"></point>
        <point x="290" y="135"></point>
        <point x="275" y="180"></point>
        <point x="260" y="195"></point>
        <point x="215" y="195"></point>
        <point x="162" y="165"></point>
      </polygon>
      <polygon color="255" filled="true" marked="false">
        <point x="150" y="255"></point>
        <point x="135" y="225"></point>
        <point x="120" y="150"></point>
        <point x="135" y="120"></point>
        <point x="150" y="105"></point>
        <point x="165" y="120"></point>
        <point x="180" y="150"></point>
        <point x="165" y="225"></point>
      </polygon>
      <circle x="135" y="90" marked="false" color="255" diameter="30" filled="true"></circle>
      <line endX="195" startY="105" marked="false" color="255" endY="60" startX="150"></line>
      <line endX="105" startY="105" marked="false" color="255" endY="60" startX="150"></line>
    </shape>
    <shape name="car" rotatable="false" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="300" y="180"></point>
        <point x="279" y="164"></point>
        <point x="261" y="144"></point>
        <point x="240" y="135"></point>
        <point x="226" y="132"></point>
        <point x="213" y="106"></point>
        <point x="203" y="84"></point>
        <point x="185" y="63"></point>
        <point x="159" y="50"></point>
        <point x="135" y="50"></point>
        <point x="75" y="60"></point>
        <point x="0" y="150"></point>
        <point x="0" y="165"></point>
        <point x="0" y="225"></point>
        <point x="300" y="225"></point>
        <point x="300" y="180"></point>
      </polygon>
      <circle x="180" y="180" marked="false" color="255" diameter="90" filled="true"></circle>
      <circle x="30" y="180" marked="false" color="255" diameter="90" filled="true"></circle>
      <polygon color="255" filled="true" marked="false">
        <point x="162" y="80"></point>
        <point x="132" y="78"></point>
        <point x="134" y="135"></point>
        <point x="209" y="135"></point>
        <point x="194" y="105"></point>
        <point x="189" y="96"></point>
        <point x="180" y="89"></point>
      </polygon>
      <circle x="47" y="195" marked="true" color="-1920102913" diameter="58" filled="true"></circle>
      <circle x="195" y="195" marked="true" color="-1920102913" diameter="58" filled="true"></circle>
    </shape>
    <shape name="circle" rotatable="false" editableColorIndex="0">
      <circle x="0" y="0" marked="true" color="-1920102913" diameter="300" filled="true"></circle>
    </shape>
    <shape name="circle 2" rotatable="false" editableColorIndex="0">
      <circle x="0" y="0" marked="true" color="-1920102913" diameter="300" filled="true"></circle>
      <circle x="30" y="30" marked="false" color="255" diameter="240" filled="true"></circle>
    </shape>
    <shape name="cow" rotatable="false" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="200" y="193"></point>
        <point x="197" y="249"></point>
        <point x="179" y="249"></point>
        <point x="177" y="196"></point>
        <point x="166" y="187"></point>
        <point x="140" y="189"></point>
        <point x="93" y="191"></point>
        <point x="78" y="179"></point>
        <point x="72" y="211"></point>
        <point x="49" y="209"></point>
        <point x="48" y="181"></point>
        <point x="37" y="149"></point>
        <point x="25" y="120"></point>
        <point x="25" y="89"></point>
        <point x="45" y="72"></point>
        <point x="103" y="84"></point>
        <point x="179" y="75"></point>
        <point x="198" y="76"></point>
        <point x="252" y="64"></point>
        <point x="272" y="81"></point>
        <point x="293" y="103"></point>
        <point x="285" y="121"></point>
        <point x="255" y="121"></point>
        <point x="242" y="118"></point>
        <point x="224" y="167"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="73" y="210"></point>
        <point x="86" y="251"></point>
        <point x="62" y="249"></point>
        <point x="48" y="208"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="25" y="114"></point>
        <point x="16" y="195"></point>
        <point x="9" y="204"></point>
        <point x="23" y="213"></point>
        <point x="25" y="200"></point>
        <point x="39" y="123"></point>
      </polygon>
    </shape>
    <shape name="cylinder" rotatable="false" editableColorIndex="0">
      <circle x="0" y="0" marked="true" color="-1920102913" diameter="300" filled="true"></circle>
    </shape>
    <shape name="dot" rotatable="false" editableColorIndex="0">
      <circle x="90" y="90" marked="true" color="-1920102913" diameter="120" filled="true"></circle>
    </shape>
    <shape name="face happy" rotatable="false" editableColorIndex="0">
      <circle x="8" y="8" marked="true" color="-1920102913" diameter="285" filled="true"></circle>
      <circle x="60" y="75" marked="false" color="255" diameter="60" filled="true"></circle>
      <circle x="180" y="75" marked="false" color="255" diameter="60" filled="true"></circle>
      <polygon color="255" filled="true" marked="false">
        <point x="150" y="255"></point>
        <point x="90" y="239"></point>
        <point x="62" y="213"></point>
        <point x="47" y="191"></point>
        <point x="67" y="179"></point>
        <point x="90" y="203"></point>
        <point x="109" y="218"></point>
        <point x="150" y="225"></point>
        <point x="192" y="218"></point>
        <point x="210" y="203"></point>
        <point x="227" y="181"></point>
        <point x="251" y="194"></point>
        <point x="236" y="217"></point>
        <point x="212" y="240"></point>
      </polygon>
    </shape>
    <shape name="face neutral" rotatable="false" editableColorIndex="0">
      <circle x="8" y="7" marked="true" color="-1920102913" diameter="285" filled="true"></circle>
      <circle x="60" y="75" marked="false" color="255" diameter="60" filled="true"></circle>
      <circle x="180" y="75" marked="false" color="255" diameter="60" filled="true"></circle>
      <rectangle endX="240" startY="195" marked="false" color="255" endY="225" startX="60" filled="true"></rectangle>
    </shape>
    <shape name="face sad" rotatable="false" editableColorIndex="0">
      <circle x="8" y="8" marked="true" color="-1920102913" diameter="285" filled="true"></circle>
      <circle x="60" y="75" marked="false" color="255" diameter="60" filled="true"></circle>
      <circle x="180" y="75" marked="false" color="255" diameter="60" filled="true"></circle>
      <polygon color="255" filled="true" marked="false">
        <point x="150" y="168"></point>
        <point x="90" y="184"></point>
        <point x="62" y="210"></point>
        <point x="47" y="232"></point>
        <point x="67" y="244"></point>
        <point x="90" y="220"></point>
        <point x="109" y="205"></point>
        <point x="150" y="198"></point>
        <point x="192" y="205"></point>
        <point x="210" y="220"></point>
        <point x="227" y="242"></point>
        <point x="251" y="229"></point>
        <point x="236" y="206"></point>
        <point x="212" y="183"></point>
      </polygon>
    </shape>
    <shape name="fish" rotatable="false" editableColorIndex="0">
      <polygon color="-1" filled="true" marked="false">
        <point x="44" y="131"></point>
        <point x="21" y="87"></point>
        <point x="15" y="86"></point>
        <point x="0" y="120"></point>
        <point x="15" y="150"></point>
        <point x="0" y="180"></point>
        <point x="13" y="214"></point>
        <point x="20" y="212"></point>
        <point x="45" y="166"></point>
      </polygon>
      <polygon color="-1" filled="true" marked="false">
        <point x="135" y="195"></point>
        <point x="119" y="235"></point>
        <point x="95" y="218"></point>
        <point x="76" y="210"></point>
        <point x="46" y="204"></point>
        <point x="60" y="165"></point>
      </polygon>
      <polygon color="-1" filled="true" marked="false">
        <point x="75" y="45"></point>
        <point x="83" y="77"></point>
        <point x="71" y="103"></point>
        <point x="86" y="114"></point>
        <point x="166" y="78"></point>
        <point x="135" y="60"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="30" y="136"></point>
        <point x="151" y="77"></point>
        <point x="226" y="81"></point>
        <point x="280" y="119"></point>
        <point x="292" y="146"></point>
        <point x="292" y="160"></point>
        <point x="287" y="170"></point>
        <point x="270" y="195"></point>
        <point x="195" y="210"></point>
        <point x="151" y="212"></point>
        <point x="30" y="166"></point>
      </polygon>
      <circle x="215" y="106" marked="false" color="255" diameter="30" filled="true"></circle>
    </shape>
    <shape name="flag" rotatable="false" editableColorIndex="0">
      <rectangle endX="75" startY="15" marked="true" color="-1920102913" endY="300" startX="60" filled="true"></rectangle>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="90" y="150"></point>
        <point x="270" y="90"></point>
        <point x="90" y="30"></point>
      </polygon>
      <line endX="90" startY="135" marked="true" color="-1920102913" endY="135" startX="75"></line>
      <line endX="90" startY="45" marked="true" color="-1920102913" endY="45" startX="75"></line>
    </shape>
    <shape name="flower" rotatable="false" editableColorIndex="0">
      <polygon color="1504722175" filled="true" marked="false">
        <point x="135" y="120"></point>
        <point x="165" y="165"></point>
        <point x="180" y="210"></point>
        <point x="180" y="240"></point>
        <point x="150" y="300"></point>
        <point x="165" y="300"></point>
        <point x="195" y="240"></point>
        <point x="195" y="195"></point>
        <point x="165" y="135"></point>
      </polygon>
      <circle x="85" y="132" marked="true" color="-1920102913" diameter="38" filled="true"></circle>
      <circle x="130" y="147" marked="true" color="-1920102913" diameter="38" filled="true"></circle>
      <circle x="192" y="85" marked="true" color="-1920102913" diameter="38" filled="true"></circle>
      <circle x="85" y="40" marked="true" color="-1920102913" diameter="38" filled="true"></circle>
      <circle x="177" y="40" marked="true" color="-1920102913" diameter="38" filled="true"></circle>
      <circle x="177" y="132" marked="true" color="-1920102913" diameter="38" filled="true"></circle>
      <circle x="70" y="85" marked="true" color="-1920102913" diameter="38" filled="true"></circle>
      <circle x="130" y="25" marked="true" color="-1920102913" diameter="38" filled="true"></circle>
      <circle x="96" y="51" marked="true" color="-1920102913" diameter="108" filled="true"></circle>
      <circle x="113" y="68" marked="false" color="255" diameter="74" filled="true"></circle>
      <polygon color="1504722175" filled="true" marked="false">
        <point x="189" y="233"></point>
        <point x="219" y="188"></point>
        <point x="249" y="173"></point>
        <point x="279" y="188"></point>
        <point x="234" y="218"></point>
      </polygon>
      <polygon color="1504722175" filled="true" marked="false">
        <point x="180" y="255"></point>
        <point x="150" y="210"></point>
        <point x="105" y="210"></point>
        <point x="75" y="240"></point>
        <point x="135" y="240"></point>
      </polygon>
    </shape>
    <shape name="house" rotatable="false" editableColorIndex="0">
      <rectangle endX="255" startY="120" marked="true" color="-1920102913" endY="285" startX="45" filled="true"></rectangle>
      <rectangle endX="180" startY="210" marked="false" color="255" endY="285" startX="120" filled="true"></rectangle>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="15" y="120"></point>
        <point x="150" y="15"></point>
        <point x="285" y="120"></point>
      </polygon>
      <line endX="270" startY="120" marked="false" color="255" endY="120" startX="30"></line>
    </shape>
    <shape name="leaf" rotatable="false" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="210"></point>
        <point x="135" y="195"></point>
        <point x="120" y="210"></point>
        <point x="60" y="210"></point>
        <point x="30" y="195"></point>
        <point x="60" y="180"></point>
        <point x="60" y="165"></point>
        <point x="15" y="135"></point>
        <point x="30" y="120"></point>
        <point x="15" y="105"></point>
        <point x="40" y="104"></point>
        <point x="45" y="90"></point>
        <point x="60" y="90"></point>
        <point x="90" y="105"></point>
        <point x="105" y="120"></point>
        <point x="120" y="120"></point>
        <point x="105" y="60"></point>
        <point x="120" y="60"></point>
        <point x="135" y="30"></point>
        <point x="150" y="15"></point>
        <point x="165" y="30"></point>
        <point x="180" y="60"></point>
        <point x="195" y="60"></point>
        <point x="180" y="120"></point>
        <point x="195" y="120"></point>
        <point x="210" y="105"></point>
        <point x="240" y="90"></point>
        <point x="255" y="90"></point>
        <point x="263" y="104"></point>
        <point x="285" y="105"></point>
        <point x="270" y="120"></point>
        <point x="285" y="135"></point>
        <point x="240" y="165"></point>
        <point x="240" y="180"></point>
        <point x="270" y="195"></point>
        <point x="240" y="210"></point>
        <point x="180" y="210"></point>
        <point x="165" y="195"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="135" y="195"></point>
        <point x="135" y="240"></point>
        <point x="120" y="255"></point>
        <point x="105" y="255"></point>
        <point x="105" y="285"></point>
        <point x="135" y="285"></point>
        <point x="165" y="240"></point>
        <point x="165" y="195"></point>
      </polygon>
    </shape>
    <shape name="line" rotatable="true" editableColorIndex="0">
      <line endX="150" startY="0" marked="true" color="-1920102913" endY="300" startX="150"></line>
    </shape>
    <shape name="line half" rotatable="true" editableColorIndex="0">
      <line endX="150" startY="0" marked="true" color="-1920102913" endY="150" startX="150"></line>
    </shape>
    <shape name="pentagon" rotatable="false" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="15"></point>
        <point x="15" y="120"></point>
        <point x="60" y="285"></point>
        <point x="240" y="285"></point>
        <point x="285" y="120"></point>
      </polygon>
    </shape>
    <shape name="person" rotatable="false" editableColorIndex="0">
      <circle x="110" y="5" marked="true" color="-1920102913" diameter="80" filled="true"></circle>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="105" y="90"></point>
        <point x="120" y="195"></point>
        <point x="90" y="285"></point>
        <point x="105" y="300"></point>
        <point x="135" y="300"></point>
        <point x="150" y="225"></point>
        <point x="165" y="300"></point>
        <point x="195" y="300"></point>
        <point x="210" y="285"></point>
        <point x="180" y="195"></point>
        <point x="195" y="90"></point>
      </polygon>
      <rectangle endX="172" startY="79" marked="true" color="-1920102913" endY="94" startX="127" filled="true"></rectangle>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="195" y="90"></point>
        <point x="240" y="150"></point>
        <point x="225" y="180"></point>
        <point x="165" y="105"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="105" y="90"></point>
        <point x="60" y="150"></point>
        <point x="75" y="180"></point>
        <point x="135" y="105"></point>
      </polygon>
    </shape>
    <shape name="person lefty" rotatable="false" editableColorIndex="0">
      <circle x="170" y="5" marked="true" color="-1920102913" diameter="80" filled="true"></circle>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="165" y="90"></point>
        <point x="180" y="195"></point>
        <point x="150" y="285"></point>
        <point x="165" y="300"></point>
        <point x="195" y="300"></point>
        <point x="210" y="225"></point>
        <point x="225" y="300"></point>
        <point x="255" y="300"></point>
        <point x="270" y="285"></point>
        <point x="240" y="195"></point>
        <point x="255" y="90"></point>
      </polygon>
      <rectangle endX="232" startY="79" marked="true" color="-1920102913" endY="94" startX="187" filled="true"></rectangle>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="255" y="90"></point>
        <point x="300" y="150"></point>
        <point x="285" y="180"></point>
        <point x="225" y="105"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="165" y="90"></point>
        <point x="120" y="150"></point>
        <point x="135" y="180"></point>
        <point x="195" y="105"></point>
      </polygon>
    </shape>
    <shape name="person righty" rotatable="false" editableColorIndex="0">
      <circle x="50" y="5" marked="true" color="-1920102913" diameter="80" filled="true"></circle>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="45" y="90"></point>
        <point x="60" y="195"></point>
        <point x="30" y="285"></point>
        <point x="45" y="300"></point>
        <point x="75" y="300"></point>
        <point x="90" y="225"></point>
        <point x="105" y="300"></point>
        <point x="135" y="300"></point>
        <point x="150" y="285"></point>
        <point x="120" y="195"></point>
        <point x="135" y="90"></point>
      </polygon>
      <rectangle endX="112" startY="79" marked="true" color="-1920102913" endY="94" startX="67" filled="true"></rectangle>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="135" y="90"></point>
        <point x="180" y="150"></point>
        <point x="165" y="180"></point>
        <point x="105" y="105"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="45" y="90"></point>
        <point x="0" y="150"></point>
        <point x="15" y="180"></point>
        <point x="75" y="105"></point>
      </polygon>
    </shape>
    <shape name="plant" rotatable="false" editableColorIndex="0">
      <rectangle endX="165" startY="90" marked="true" color="-1920102913" endY="300" startX="135" filled="true"></rectangle>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="135" y="255"></point>
        <point x="90" y="210"></point>
        <point x="45" y="195"></point>
        <point x="75" y="255"></point>
        <point x="135" y="285"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="165" y="255"></point>
        <point x="210" y="210"></point>
        <point x="255" y="195"></point>
        <point x="225" y="255"></point>
        <point x="165" y="285"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="135" y="180"></point>
        <point x="90" y="135"></point>
        <point x="45" y="120"></point>
        <point x="75" y="180"></point>
        <point x="135" y="210"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="165" y="180"></point>
        <point x="165" y="210"></point>
        <point x="225" y="180"></point>
        <point x="255" y="120"></point>
        <point x="210" y="135"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="135" y="105"></point>
        <point x="90" y="60"></point>
        <point x="45" y="45"></point>
        <point x="75" y="105"></point>
        <point x="135" y="135"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="165" y="105"></point>
        <point x="165" y="135"></point>
        <point x="225" y="105"></point>
        <point x="255" y="45"></point>
        <point x="210" y="60"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="135" y="90"></point>
        <point x="120" y="45"></point>
        <point x="150" y="15"></point>
        <point x="180" y="45"></point>
        <point x="165" y="90"></point>
      </polygon>
    </shape>
    <shape name="square" rotatable="false" editableColorIndex="0">
      <rectangle endX="270" startY="30" marked="true" color="-1920102913" endY="270" startX="30" filled="true"></rectangle>
    </shape>
    <shape name="square 2" rotatable="false" editableColorIndex="0">
      <rectangle endX="270" startY="30" marked="true" color="-1920102913" endY="270" startX="30" filled="true"></rectangle>
      <rectangle endX="240" startY="60" marked="false" color="255" endY="240" startX="60" filled="true"></rectangle>
    </shape>
    <shape name="star" rotatable="false" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="151" y="1"></point>
        <point x="185" y="108"></point>
        <point x="298" y="108"></point>
        <point x="207" y="175"></point>
        <point x="242" y="282"></point>
        <point x="151" y="216"></point>
        <point x="59" y="282"></point>
        <point x="94" y="175"></point>
        <point x="3" y="108"></point>
        <point x="116" y="108"></point>
      </polygon>
    </shape>
    <shape name="target" rotatable="false" editableColorIndex="0">
      <circle x="0" y="0" marked="true" color="-1920102913" diameter="300" filled="true"></circle>
      <circle x="30" y="30" marked="false" color="255" diameter="240" filled="true"></circle>
      <circle x="60" y="60" marked="true" color="-1920102913" diameter="180" filled="true"></circle>
      <circle x="90" y="90" marked="false" color="255" diameter="120" filled="true"></circle>
      <circle x="120" y="120" marked="true" color="-1920102913" diameter="60" filled="true"></circle>
    </shape>
    <shape name="tree" rotatable="false" editableColorIndex="0">
      <circle x="118" y="3" marked="true" color="-1920102913" diameter="94" filled="true"></circle>
      <rectangle endX="180" startY="195" marked="false" color="-1653716737" endY="300" startX="120" filled="true"></rectangle>
      <circle x="65" y="21" marked="true" color="-1920102913" diameter="108" filled="true"></circle>
      <circle x="116" y="41" marked="true" color="-1920102913" diameter="127" filled="true"></circle>
      <circle x="45" y="90" marked="true" color="-1920102913" diameter="120" filled="true"></circle>
      <circle x="104" y="74" marked="true" color="-1920102913" diameter="152" filled="true"></circle>
    </shape>
    <shape name="triangle" rotatable="false" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="30"></point>
        <point x="15" y="255"></point>
        <point x="285" y="255"></point>
      </polygon>
    </shape>
    <shape name="triangle 2" rotatable="false" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="30"></point>
        <point x="15" y="255"></point>
        <point x="285" y="255"></point>
      </polygon>
      <polygon color="255" filled="true" marked="false">
        <point x="151" y="99"></point>
        <point x="225" y="223"></point>
        <point x="75" y="224"></point>
      </polygon>
    </shape>
    <shape name="truck" rotatable="false" editableColorIndex="0">
      <rectangle endX="195" startY="45" marked="true" color="-1920102913" endY="187" startX="4" filled="true"></rectangle>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="296" y="193"></point>
        <point x="296" y="150"></point>
        <point x="259" y="134"></point>
        <point x="244" y="104"></point>
        <point x="208" y="104"></point>
        <point x="207" y="194"></point>
      </polygon>
      <rectangle endX="195" startY="60" marked="false" color="-1" endY="105" startX="195" filled="true"></rectangle>
      <polygon color="255" filled="true" marked="false">
        <point x="238" y="112"></point>
        <point x="252" y="141"></point>
        <point x="219" y="141"></point>
        <point x="218" y="112"></point>
      </polygon>
      <circle x="234" y="174" marked="false" color="255" diameter="42" filled="true"></circle>
      <rectangle endX="214" startY="185" marked="true" color="-1920102913" endY="194" startX="181" filled="true"></rectangle>
      <circle x="144" y="174" marked="false" color="255" diameter="42" filled="true"></circle>
      <circle x="24" y="174" marked="false" color="255" diameter="42" filled="true"></circle>
      <circle x="24" y="174" marked="true" color="-1920102913" diameter="42" filled="false"></circle>
      <circle x="144" y="174" marked="true" color="-1920102913" diameter="42" filled="false"></circle>
      <circle x="234" y="174" marked="true" color="-1920102913" diameter="42" filled="false"></circle>
    </shape>
    <shape name="turtle" rotatable="true" editableColorIndex="0">
      <polygon color="1504722175" filled="true" marked="false">
        <point x="215" y="204"></point>
        <point x="240" y="233"></point>
        <point x="246" y="254"></point>
        <point x="228" y="266"></point>
        <point x="215" y="252"></point>
        <point x="193" y="210"></point>
      </polygon>
      <polygon color="1504722175" filled="true" marked="false">
        <point x="195" y="90"></point>
        <point x="225" y="75"></point>
        <point x="245" y="75"></point>
        <point x="260" y="89"></point>
        <point x="269" y="108"></point>
        <point x="261" y="124"></point>
        <point x="240" y="105"></point>
        <point x="225" y="105"></point>
        <point x="210" y="105"></point>
      </polygon>
      <polygon color="1504722175" filled="true" marked="false">
        <point x="105" y="90"></point>
        <point x="75" y="75"></point>
        <point x="55" y="75"></point>
        <point x="40" y="89"></point>
        <point x="31" y="108"></point>
        <point x="39" y="124"></point>
        <point x="60" y="105"></point>
        <point x="75" y="105"></point>
        <point x="90" y="105"></point>
      </polygon>
      <polygon color="1504722175" filled="true" marked="false">
        <point x="132" y="85"></point>
        <point x="134" y="64"></point>
        <point x="107" y="51"></point>
        <point x="108" y="17"></point>
        <point x="150" y="2"></point>
        <point x="192" y="18"></point>
        <point x="192" y="52"></point>
        <point x="169" y="65"></point>
        <point x="172" y="87"></point>
      </polygon>
      <polygon color="1504722175" filled="true" marked="false">
        <point x="85" y="204"></point>
        <point x="60" y="233"></point>
        <point x="54" y="254"></point>
        <point x="72" y="266"></point>
        <point x="85" y="252"></point>
        <point x="107" y="210"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="119" y="75"></point>
        <point x="179" y="75"></point>
        <point x="209" y="101"></point>
        <point x="224" y="135"></point>
        <point x="220" y="225"></point>
        <point x="175" y="261"></point>
        <point x="128" y="261"></point>
        <point x="81" y="224"></point>
        <point x="74" y="135"></point>
        <point x="88" y="99"></point>
      </polygon>
    </shape>
    <shape name="wheel" rotatable="false" editableColorIndex="0">
      <circle x="3" y="3" marked="true" color="-1920102913" diameter="294" filled="true"></circle>
      <circle x="30" y="30" marked="false" color="255" diameter="240" filled="true"></circle>
      <line endX="150" startY="285" marked="true" color="-1920102913" endY="15" startX="150"></line>
      <line endX="285" startY="150" marked="true" color="-1920102913" endY="150" startX="15"></line>
      <circle x="120" y="120" marked="true" color="-1920102913" diameter="60" filled="true"></circle>
      <line endX="79" startY="40" marked="true" color="-1920102913" endY="269" startX="216"></line>
      <line endX="269" startY="84" marked="true" color="-1920102913" endY="221" startX="40"></line>
      <line endX="269" startY="216" marked="true" color="-1920102913" endY="79" startX="40"></line>
      <line endX="221" startY="40" marked="true" color="-1920102913" endY="269" startX="84"></line>
    </shape>
    <shape name="x" rotatable="false" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="270" y="75"></point>
        <point x="225" y="30"></point>
        <point x="30" y="225"></point>
        <point x="75" y="270"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="30" y="75"></point>
        <point x="75" y="30"></point>
        <point x="270" y="225"></point>
        <point x="225" y="270"></point>
      </polygon>
    </shape>
  </turtleShapes>
  <linkShapes>
    <shape name="default" curviness="0.0">
      <lines>
        <line x="-0.2" visible="false">
          <dash value="0.0"></dash>
          <dash value="1.0"></dash>
        </line>
        <line x="0.0" visible="true">
          <dash value="1.0"></dash>
          <dash value="0.0"></dash>
        </line>
        <line x="0.2" visible="false">
          <dash value="0.0"></dash>
          <dash value="1.0"></dash>
        </line>
      </lines>
      <indicator>
        <shape name="link direction" rotatable="true" editableColorIndex="0">
          <line endX="90" startY="150" marked="true" color="-1920102913" endY="180" startX="150"></line>
          <line endX="210" startY="150" marked="true" color="-1920102913" endY="180" startX="150"></line>
        </shape>
      </indicator>
    </shape>
  </linkShapes>
  <previewCommands>setup repeat 75 [ go ]</previewCommands>
</model>
